// Livre "Construire un annuaire d'entreprise avec LDAP"
// Auteur : Marcel Rizcallah
//
// Application Java/Swing testee avec Sun JDK 1.2.2
// sur un annuaire LDAP Netscape Directory 4.1 sous Windows 2000
//
//							(o=lamine.org)
//				+-------------------------------|-----------------+-------------------------+
//			(ou=Business Unit) 				(ou=Direction Generale) 	(ou=Groupes)
//			+-----------|---------------+	
//	(cn=Transport Tourisme Loisir)		(cn=Banque)
//                  |
//            uid=LCHENTOUF 
//
// Compilation : %JAVA_HOME%\bin\javac LC_JNDI.java
// Execution   : %JAVA_HOME%\bin\java  LC_JNDI

// Une classe utilitaire tres pratique en Java (notamment pour les vecteurs)
import java.util.* ;

// Les classes IHM de Java dont le standard AWT et l extension JFC/Swing
// pour un design original
import java.awt.* ;
import java.awt.event.* ;
import javax.swing.* ;
import javax.swing.tree.* ;
import javax.swing.table.* ;
import javax.swing.event.* ;

// Les classes JNDI pour l acces aux services d annuaires
// JNDI propose un modele hierarchique presentant une abtsraction forte
// lui permettant de coupler a de multiples services tels que LDAP, DNS, NIS ...
// La variable d environnement CLASSPATH doit pointer obligatoirement :
// 	%JAVA_HOME%\lib\JNDI.JAR;
// 	%JAVA_HOME%\lib\JAAS.JAR
// 	%JAVA_HOME%\lib\LDAP.JAR;
// 	%JAVA_HOME%\lib\LDAPBP.JAR;
// 	%JAVA_HOME%\lib\PROVIDERUTIL.JAR;
//
// Correspondance JNDI<->LDAP
// bind 	= InitialDirContext 
//		  en prenant les informations depuis les proprietes de environment (Context.addToEnvironment() 
//		  et Context.removeFromEnvironment())  
// unbind       = Context.close() 
//		  Cela ne libre que certaines ressources utilisees par le contexte, et non pas toutes car elles peuvent
//		  etre partagees par d autres types de contexte (DNS par exemple). La fonction unbind specifiee dans le 
//		  protocole LDAP libere quant a elle completement la session. Il faut fermer tous les contextes pour etre
//		  sur de liberer toutes les ressources
// search       = DirContext.search() 
//		  La methode existe sous plusieurs polymorphismes suivant qu on integre un filtre LDAP ou pas etc...
// compare      = DirContext.search() 
//                Par souci de performence, on peut faire une recherche sans attribut a retourner
// modify  	= DirContext.modifyAttributes() 
//		  Un de ses parametres contient pour chaque attribut la valeur et le type d operation (modification,
//		  suppression, ajout)
// abandon      = Context.close() ou NamingEnumeration.close()  
//                La fermeture d un contexte arrete les operations en cours, tout comme la fermeture de l objet resultat
//
import javax.naming.*;
import javax.naming.directory.*;

// L application LC_JNDI herite d une JFrame, ses methodes et proprietes
public class LC_JNDI 
	extends JFrame 
	implements TreeSelectionListener
{ 
	// Qq constantes
	final String attributLibelle = "Attribut=" ;
	final String valeurLibelle   = "Valeur=" ;
	final String attributIcone   = "attribut.gif" ;
	final String valeurIcone     = "valeur.gif" ;
	
	// Chaines de connexion LDAP
	String urlLDAP  = "pentium.lamine.org";
	String portLDAP = "389" ;
	String baseLDAP = "o=lamine.org";
	
	// Contexte LDAP
	DirContext contexteLDAP;
	
	// Quelques fenetres
	JDesktopPane panneauFenetre ;
	JInternalFrame fenetreConnexion;
	JInternalFrame fenetreArbre;
	boolean ouvrirTableau ;
	
	// Un arbre qui va representer la hierarchie LDAP
	JTree arbre ;
	DefaultMutableTreeNode feuille;	

	// L objet suivant contiendra tous les messages generes par l application
	JTextArea log ;
	
	// L application se lance via la ligne de commande : java LC_JNDI
	// Il est possible de la transformer en applet
	public static void main (String args[])
	{
		LC_JNDI lC_Jndi = new LC_JNDI(640,480,"Exemple LDAP avec JNDI");
	}

	// Constructeur de l application
	// Il s agit d une fenetre JFrame
	public LC_JNDI(int xFenetre, int yFenetre, String titreFenetre) 
	{
		try
		{
			setTitle(titreFenetre) ;
			// Surcharge de l evenement de fermeture de la fenetre 
			addWindowListener(new WindowAdapter() 
						{
        						public void windowClosing(WindowEvent evennement) 
        						{
								try
								{
					        			// Liberation des ressources utilisees par le contexte LDAP
					        			if (contexteLDAP != null) contexteLDAP.close() ;
					        		}
								catch (NamingException erreur)
	    							{
	    									ecrireLog("!! Erreur Listing : "+erreur) ;
	    							}	    
        							System.exit(0);
        						}
    						});

			setSize(xFenetre,yFenetre) ;

			// Creation du support des sous fenetres
			panneauFenetre = new JDesktopPane() ;
			setContentPane(panneauFenetre) ;

			// Creation de la fenetre de connexion LDAP
			panneauFenetre.add(creerFenetreConnexion())	;

			// Creation de la fenetre de log
			panneauFenetre.add(creerFenetreLog(),2)	;

			setVisible(true) ;
		}			
		catch (Exception erreur)
		{
			// Gerer les erreurs d initialisation
			ecrireLog("!! Erreur dans le constructeur : "+erreur) ;
		}
	}

	// Sous fenetre contenant les logs
	public JInternalFrame creerFenetreLog()
	{
		JInternalFrame fenetre = new JInternalFrame("Log",true,false,true,true) ; 
		try
		{
			fenetre.setSelected(true) ;
			fenetre.setSize(600,300) ;
			fenetre.setLocation(30,80);
			fenetre.setVisible(true) ;
						
			//log = new JTextPane() ;
			log = new JTextArea() ;
			log.setEditable(false);
			log.setLineWrap(false);
			log.setBackground(java.awt.Color.black);
			log.setForeground(java.awt.Color.yellow);
			fenetre.setContentPane(new JScrollPane(log));
		}			
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Creation Fenetre Login : "+erreur) ;
		}
		return fenetre ;							
	}
	
	// Ecriture d un message dans un log
	public void ecrireLog(String texte)
	{
		System.out.println(texte);	
		log.setText(log.getText()+texte+"\n");
		if (texte.indexOf("!!")==0) JOptionPane.showMessageDialog(this,texte,"Erreur detectee",JOptionPane.ERROR_MESSAGE);
	}

	// Sous fenetre contenant les parametres de connexion LDAP	
	public JInternalFrame creerFenetreConnexion()
	{
		fenetreConnexion = new JInternalFrame("Connexion a l'annuaire LDAP",false,false,false,false) ; 
		try
		{
			fenetreConnexion.setSelected(true) ;
			fenetreConnexion.setVisible(true) ;
			fenetreConnexion.setSize(250,175) ;

			JPanel panneauLibelle = new JPanel();
			panneauLibelle.setLayout(new GridLayout(0,1));
			panneauLibelle.add(new JLabel("Serveur LDAP"));
			panneauLibelle.add(new JLabel("Port LDAP"));
			panneauLibelle.add(new JLabel("Base LDAP"));
			panneauLibelle.add(new JLabel("Login"));
			panneauLibelle.add(new JLabel("Mot de passe"));

			JPanel panneauSaisie = new JPanel();
			panneauSaisie.setLayout(new GridLayout(0,1));
			final JTextField urlLdapConnexion      = new JTextField(urlLDAP,15) ;
			final JTextField portLdapConnexion     = new JTextField(portLDAP,15) ;
			final JTextField baseLdapConnexion     = new JTextField(baseLDAP,15) ;
			final JTextField loginConnexion        = new JTextField("CHENTOUF Lamine",15) ;
			final JPasswordField motpasseConnexion = new JPasswordField("animal00",15) ;
			panneauSaisie.add(urlLdapConnexion) ;
			panneauSaisie.add(portLdapConnexion) ;
			panneauSaisie.add(baseLdapConnexion) ;
			panneauSaisie.add(loginConnexion) ;
			panneauSaisie.add(motpasseConnexion) ;

			// Bouton de connexion
			JButton boutonOk = new JButton("Se Connecter") ;

			JPanel panneauConnexion = new JPanel() ;
			panneauConnexion.setLayout(new BorderLayout());
			panneauConnexion.add(panneauLibelle,BorderLayout.WEST);
			panneauConnexion.add(panneauSaisie, BorderLayout.CENTER);
			panneauConnexion.add(boutonOk,      BorderLayout.SOUTH) ;
			fenetreConnexion.setContentPane(panneauConnexion);

			boutonOk.addActionListener(new ActionListener()
							{
								public void actionPerformed(ActionEvent evennement)
								{
									urlLDAP  = urlLdapConnexion.getText() ;
									portLDAP = portLdapConnexion.getText();
									baseLDAP = baseLdapConnexion.getText();
									try
									{
										// On capture le login et mot de passe pour declencher la connexion LDAP
										boutonConnexion(loginConnexion.getText(),new String(motpasseConnexion.getPassword())) ;
									}
									catch(Exception erreur)
									{
										ecrireLog("!! Erreur Bouton OK : "+erreur) ;
									}
								}
							});
			
		}			
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Creation Fenetre Login : "+erreur) ;
		}
		return fenetreConnexion ;							
	}

	// Sous fenetre Tableau
	public JInternalFrame creerFenetreTableau(String dnLDAP, Vector lignes, Vector colonnes)
	{
		final JInternalFrame fenetre = new JInternalFrame(dnLDAP,true,false,false,false) ; 
		final String DnLDAP = dnLDAP ;
		try
		{
			final JTable tableau  = new JTable(lignes,colonnes) ;
			JButton boutonMaj = new JButton("Mettre  jour dans l annuaire LDAP") ;

			JPanel panneau = new JPanel() ;
			fenetre.setContentPane(panneau);
			panneau.setLayout(new BorderLayout());
			panneau.add(new JScrollPane(tableau),BorderLayout.CENTER) ;
			panneau.add(boutonMaj, BorderLayout.SOUTH) ;

			fenetre.setSelected(true) ;
			fenetre.setSize(600,100) ;

			fenetreArbre.setVisible(false) ;
			fenetre.setVisible(true) ;
			ouvrirTableau = false ;			
			boutonMaj.addActionListener(new ActionListener()
							{
								public void actionPerformed(ActionEvent evennement)
								{
									try
									{
										// Cela va declencher la maj avec les infos du tableau
										boutonMaj(fenetre,tableau) ;
									}
									catch(Exception erreur)
									{
										ecrireLog("!! Erreur Bouton OK : "+erreur) ;
									}
								}
							});
		}			
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Creation Fenetre Tableau : "+erreur) ;
		}
		return fenetre ;							
	}

	// Sous fenetre des arbres qui va representer le modele LDAP
	public JInternalFrame creerFenetreArbre()
	{
		fenetreArbre = new JInternalFrame("Arbre LDAP",true,false,true,true) ; 
		try
		{
			fenetreArbre.setSelected(true) ;
			fenetreArbre.setVisible(true) ;
			fenetreArbre.setSize(600,300) ;
			fenetreArbre.setLocation(30,30);

			feuille = new DefaultMutableTreeNode(baseLDAP) ; 
			arbre = new JTree(feuille) ; 
			arbre.putClientProperty("JTree.lineStyle","Angled"); 
			arbre.addTreeSelectionListener(this) ; 
			arbre.setCellRenderer(new arbreIHM());

			fenetreArbre.setContentPane(new JScrollPane(arbre)) ;
		}			
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Creation Fenetre Arbre : "+erreur) ;
		}
		return fenetreArbre ;							
	}

    	// On surcharge des methodes de rendu de l arbre pour imposer des icones suivant qu on est 
    	// sur une feuille ou un attribut
    	class arbreIHM extends DefaultTreeCellRenderer 
    	{
    		ImageIcon imageAttribut;
    		ImageIcon imageValeur;

	    	public arbreIHM() 
    		{
        		imageAttribut = new ImageIcon(attributIcone);
        		imageValeur   = new ImageIcon(valeurIcone);
    		}

    		public Component getTreeCellRendererComponent(JTree arbre,Object objet,boolean selectionner,boolean derouler,boolean fils,int ligne,boolean actif) 
    		{
	        	super.getTreeCellRendererComponent(arbre,objet,selectionner,derouler,fils,ligne,actif);
	        	DefaultMutableTreeNode feuille = (DefaultMutableTreeNode)objet ;
	        	String libelle = (String)feuille.getUserObject() ;
			if (libelle.indexOf(attributLibelle)==0) setIcon(imageAttribut);
			else if (libelle.indexOf(valeurLibelle)==0) setIcon(imageValeur);
	 	      	return this;
    		}
	}	    	

	// Analyse du comportement de l arbre
	public void valueChanged(TreeSelectionEvent evennement)
	{
		try
		{
		        // Test de precaution
 		        feuille = (DefaultMutableTreeNode) evennement.getPath().getLastPathComponent();
		        if (feuille == null)   return;
		        String feuilleLibelle = evennement.getPath().getLastPathComponent().toString() ;
			if (feuilleLibelle.indexOf(valeurLibelle) == 0 || feuilleLibelle.indexOf(attributLibelle) == 0)  return;
			if (feuille.isLeaf()) rafraichirArbre() ;
			// A t on affaire a un user ?
			else if (!feuilleLibelle.equals(baseLDAP) && comparerLDAP(feuilleLibelle))
			{
				feuille.removeAllChildren() ;
				DefaultTreeModel modeleArbre = (DefaultTreeModel)arbre.getModel(); 
				modeleArbre.nodeChanged((DefaultMutableTreeNode)evennement.getPath().getPathComponent(0)); 
				arbre.revalidate(); 
				arbre.getParent().repaint();				
	    			ouvrirTableau = true ;
				rafraichirArbre() ;
			}				
		}
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Click Arbre : "+erreur) ;
		}
	}

	// Recherche LDAP executee apres click sur une feuille de l arbre
	public void rafraichirArbre()
	{
		// On liste le contenu du DN selectionne
		listerDnLDAP(feuille);
		
		// On va afficher les valeurs des attributs
		// String[] attributRechercherLDAP = {"givenName","sn"};
		String[] attributRechercherLDAP = {"objectclass","givenName","sn","telephonenumber","mail","facsimiletelephonenumber"} ;
		
		rechercherLDAP(attributRechercherLDAP,feuille);
	}

	// Action du bouton de connexion
	public void boutonConnexion(String loginConnexion, String motpasseConnexion)
	{
		// On tente de se connecter
		contexteLDAP = initialiserConnexionLDAP(preparerEnvironnement(loginConnexion,motpasseConnexion)) ;
		if (contexteLDAP == null) return ;
		// Creation de la fenetre des arbres
		panneauFenetre.add(creerFenetreArbre())	;

		// On cree un arbre listant les objects de type organizationalUnit dans l arbre
		//String[] attributRechercherLDAP = {"description"};
		String[] attributRechercherLDAP =  {"cn","description"};
		lireDnArbreLDAP(arbreLDAP(attributRechercherLDAP,"(objectclass=organizationalUnit)"));
	}
	
	// Action du bouton de MAJ
	public void boutonMaj(JInternalFrame fenetre, JTable tableau)
	{
		try
		{
			//String[][] attributMajLDAP = {{"telephonenumber","01.02.03.04.05.06"}, 
			//			      {"mail","lchentouf@valoris.com"},
			//			      {"givenName","Lamine"},
			//			      {"sn","CHENTOUF"}
			//			     };
			String[][] attributMajLDAP = new String [tableau.getColumnCount()][2];


			// Le tableau est lu pour capturer les noms des colonnes (noms des attributs LDAP)
			// et la ligne de donnees (valeur des attributs LDAP)			
			for (int i=0 ; i < tableau.getColumnCount(); i++)
			{
				attributMajLDAP[i][0] = tableau.getColumnName(i);
				attributMajLDAP[i][1] = (String)tableau.getValueAt(0,i) ;	
			}
			modifierLDAP(fenetre.getTitle(), attributMajLDAP) ;
			fenetre.dispose();
			fenetreArbre.setVisible(true);
			ouvrirTableau = false ;
			rafraichirArbre();
		}			
		catch (Exception erreur)
		{
			ecrireLog("!! Erreur Click Maj : "+erreur) ;
		}
	}

	// Methode de lecture de resultalt LDAP de l arbre sous une forme NamingEnumeration
	public void lireDnArbreLDAP(NamingEnumeration dnLdapListe)
	{
        try
        {
			while(dnLdapListe!=null && dnLdapListe.hasMore())	
			{
				// Une requete peut retourner plusieurs elements
				// On boucle sur un curseur tant qu il y des DN a lire
				SearchResult dnLDAP = (SearchResult)dnLdapListe.next();
				DefaultMutableTreeNode feuilleDn = new DefaultMutableTreeNode(dnLDAP.getName());
				feuille.add(feuilleDn) ;

				ecrireLog("\t@@ DN = "+dnLDAP.getName());
				// On recupere les attributs et on les analyse
				Attributes attributListe   = dnLDAP.getAttributes();
				lireAttributLDAP(dnLDAP.getName(),attributListe,feuilleDn);
			}
	    }	
	    catch (NamingException erreur)
	    {
	    	ecrireLog("!! Erreur Desassemblage : "+erreur) ;
	    }	    

	}

	// Methode de lecture d attribut LDAP
	// On visualisera ses valeurs a moins que nous specificons dans l environnement
	// la propriete nom d attribut seulement (environnementLDAP.put("java.naming.ldap.typesOnly","true"))
	// Les valeurs seront igores lors de l appel des methodes DirContext.getAttributes() et DirContext.search()
	// De meme la resolution des alias peut etre ingoree ou non via la propriete java.naming.ldap.derefAliases de environnementLDAP
	// avec les valeurs always (defaut), never, finding, searching

	public void lireAttributLDAP(String dnLDAP, Attributes attributListe,DefaultMutableTreeNode feuilleDn)
	{
	    Vector colonnes = new Vector() ;
	    Vector lignes   = new Vector() ;
            try
            {
		if (attributListe != null)
		{
			Vector ligne    = new Vector() ;
			lignes.addElement(ligne) ;
			for (NamingEnumeration attributElement = attributListe.getAll();attributElement.hasMoreElements();)
			{
				// On boucle sur tous les attributs de l element trouve
				Attribute attribut = (Attribute)attributElement.next();
				// On obtient le nom de l attribut (par exemple sn ou cn)
				String libelle = attribut.getID() ;
				ecrireLog("\t\t"+attributLibelle+libelle);
				DefaultMutableTreeNode feuilleAttribut = new DefaultMutableTreeNode(attributLibelle+libelle);
				feuilleDn.add(feuilleAttribut) ;
				// Il faut lire les valeurs de chacun de ces attributs, sachant
				// qu ils peuvent etre mono ou multi values
				String valeur = null ;
				for (Enumeration valeurListe = attribut.getAll() ; valeurListe.hasMoreElements();)
				{
					valeur = (String)valeurListe.nextElement() ;
					ecrireLog("\t\t\t"+valeurLibelle+valeur) ;
					feuilleAttribut.add(new DefaultMutableTreeNode(valeurLibelle+valeur));
				}
				if (libelle.equals("givenName") || libelle.equals("sn") || libelle.equals("telephonenumber") || libelle.equals("mail")) 
				{
					colonnes.addElement(libelle);				
					ligne.addElement(valeur) ;
				}
			}
			// Creation de la fenetre du tableau
			if (ouvrirTableau) panneauFenetre.add(creerFenetreTableau(dnLDAP,lignes,colonnes))	;
		}
	    }	
	    catch (NamingException erreur)
	    {
	    	ecrireLog("!! Erreur Desassemblage : "+erreur) ;
	    }	    

	}

	// On parametre l environnement du contexte LDAP
	// Il s agit la d un parametrage de l authentification
	public Hashtable preparerEnvironnement(String loginConnexion,String motpasseConnexion)
	{

		// Preparation de l environnement
		Hashtable environnementLDAP = new Hashtable();
		// Le contexte LDAP est selectionne en particulier
		environnementLDAP.put(Context.INITIAL_CONTEXT_FACTORY,"com.sun.jndi.ldap.LdapCtxFactory");
		// Adresse du serveur LDAP
		environnementLDAP.put(Context.PROVIDER_URL,"ldap://"+urlLDAP+":"+portLDAP+"/"+baseLDAP);
		// On peut forcer la version du protocole LDAP a utiliser
		environnementLDAP.put("java.naming.ldap.version","3");
		// On peut indiquer s il faut ignorer les alias (ignore), les suivre automatiquement(follow)
		// ou les exploiter pas a pas (throw) en interceptant les exceptions ReferralException
	        environnementLDAP.put(Context.REFERRAL,"ignore");
		// Mode d authentification:  none (anonyme), simple (mot de passe circulant en clair)
		//environnementLDAP.put(Context.SECURITY_AUTHENTICATION,"none");
		// ou specifier un mecanisme plus complexe SASL introduit dans le protocole LDAP v3 
		// par exemple CRAM-MD5 pour Netscape ou Innosoft
		environnementLDAP.put(Context.SECURITY_AUTHENTICATION,"simple");
		// Login et mot de passe de connexion requis sauf si connexion anonyme
		environnementLDAP.put(Context.SECURITY_PRINCIPAL,"cn="+loginConnexion);
		environnementLDAP.put(Context.SECURITY_CREDENTIALS,motpasseConnexion);
		return environnementLDAP ;
	}

	// Equivalent du bind du protocole LDAP
	public DirContext initialiserConnexionLDAP(Hashtable environnementLDAP)
	{
		contexteLDAP = null ;
		try
		{
			// Tentative de connexion, equivalent du bind
			// Les parametres de connexion sont donnes via l environnement LDAP
			contexteLDAP = new InitialDirContext(environnementLDAP);
			// Ces parametres peuvent etre changes a nouveau avec la methode contexteLDAP.addToEnvironment()
			ecrireLog("-- Connexion acceptee") ;
			JOptionPane.showMessageDialog(this,"Bienvenu "+((environnementLDAP.get(Context.SECURITY_PRINCIPAL).equals("cn="))?"Anonyme":environnementLDAP.get(Context.SECURITY_PRINCIPAL)),"Bonjour",JOptionPane.INFORMATION_MESSAGE);    
			fenetreConnexion.dispose();
		}			
		catch (Exception erreur)
		{
			// En cas de mauvais mot de passe par exemple, les erreurs AuthenticationException
			// ou AuthenticationNotSupportedException peuvent tre interceptes ici
			// selon que le mode d'authentification du serveur LDAP cible
			ecrireLog("!! Erreur Authentification : "+erreur) ;
		}
		return contexteLDAP ;
	}

	// Recherche une liste de DN, equivalent de search de LDAP
	// notamment qd on deroule l arbre
	public void listerDnLDAP(DefaultMutableTreeNode feuilleDn)
	{
	    if (contexteLDAP==null) return ;
	    NamingEnumeration dnLdapListe = null ;
	    try
	    {
		String dnLDAP = (String)feuilleDn.getUserObject() ;
		ecrireLog("-- Liste des DN sous "+dnLDAP);
		// Recherche sous le DN
	    	dnLdapListe = contexteLDAP.list(dnLDAP); 
	    	// Cette methode applique aussi sur un schema LDAP obtenu via DirContext schemaLDAP = contexteLDAP.getSchema("ou=Business Unit")
	    	// Analyse du resultalt
	        while (dnLdapListe.hasMore()) 
	        {
	        	String libelle = ((NameClassPair)dnLdapListe.next()).getName() ;
			ecrireLog("\t@@ "+libelle);
			if (feuilleDn != null) feuilleDn.add(new DefaultMutableTreeNode(libelle+","+dnLDAP)) ;
	        }
	    }
	    catch (NamingException erreur)
	    {
	    	ecrireLog("!! Erreur Listing : "+erreur) ;
	    }	    
	}

	// Recherche simple d attributs via un filtre
	// arbreLDAP est appele a la connexion
	public NamingEnumeration arbreLDAP(String[] attributRechercherLDAP,String filtreLDAP)
	{
	    NamingEnumeration dnLdapListe = null ;
	    if (contexteLDAP==null) return dnLdapListe;
	    
	    // Rechercher par exemple le nom, le telephone et l email
	    // String[] attributRechercherLDAP = {"sn","telephonenumber","mail"};
	    // pour les personnes de la famille CHENTOUF ayant un email
	    // String filtreLDAP = "(&(sn=CHENTOUF)(mail=*))";

	    // On peut limiter les resultats de la recherche	
	    SearchControls limitationLDAP = new SearchControls();
	    // On ne retourne que les attributs demandes
	    limitationLDAP.setReturningAttributes(attributRechercherLDAP);
	    // pour une profondeur specifique
	    limitationLDAP.setSearchScope(SearchControls.SUBTREE_SCOPE);
	    // Si on ne veut retourner qu une seule ligne :
	    //limitationLDAP.setCountLimit(1);
	    // Si on veut limiter dans le temps d execution, par exemple 10s
	    //limitationLDAP.setCountLimit(10000);

	    // Recherche des elements compte tenu des limitations
	    try
	    {
		ecrireLog("-- Arbre LDAP "+filtreLDAP);
	    	dnLdapListe = contexteLDAP.search("",filtreLDAP,limitationLDAP);
	    }
	    // Il faut penser a intercepter les alias si on les traite manuellement environnementLDAP.put(Context.REFERRAL,"throw")	
	    // catch(ReferralException erreur)
	    // {		    
	    //	ecrireLog("Alias trouve");
	    //}
	    catch (NamingException erreur)
	    {
	    	ecrireLog("!! Erreur Arbre LDAP : "+erreur) ;
	    }	    
 	    return dnLdapListe;
	}

	// Lecture des valeurs d un attribut particulier via son dn
	// Appelee via l arbre pour eventuellement donner un tableau simplifie attribut-valeur
	public void rechercherLDAP(String[] attributRechercherLDAP,DefaultMutableTreeNode feuilleDn)
	{
	    if (contexteLDAP==null) return ;		
	    Attributes attributLDAP = null ;
	    
	    // Rechercher par exemple cn=Transport Tourisme Loisir, ou=Business Unit
	    try
	    {
	    	String dnLDAP = (String)feuilleDn.getUserObject() ;
		ecrireLog("-- Lecture du DN ("+dnLDAP+")");
	    	attributLDAP = contexteLDAP.getAttributes(dnLDAP,attributRechercherLDAP);
		lireAttributLDAP(dnLDAP,attributLDAP,feuilleDn);
	    }
	    catch (NamingException erreur)
	    {
	    	ecrireLog("!! Erreur Recherche DN : "+erreur) ;
	    }	    
	}

	// Operation de comparaison
	// Appelee depuis arbre pour determiner si c est un DN de type inetOrgPerson
	public boolean comparerLDAP(String dnLDAP)
	{
		boolean trouver=false ;
		try
		{
			// On limite la recherche a l objet lui meme
			SearchControls limitationLDAP = new SearchControls();
			limitationLDAP.setSearchScope(SearchControls.OBJECT_SCOPE) ;
			// et on n a pas besoin de retourner les attributs
		        limitationLDAP.setReturningAttributes(null);
 	        	NamingEnumeration comparaisonLDAP = contexteLDAP.search(dnLDAP,"(objectclass=inetOrgPerson)", limitationLDAP);
 	        	// Le contenu du resultat determine si on a trouve ou non
 	        	if (comparaisonLDAP != null && comparaisonLDAP.hasMoreElements()) 
 	        	{
 	        		ecrireLog("-- Personne detectee sous "+dnLDAP);
 	        		trouver=true ;
			} 	        		
		}
	    	catch (NamingException erreur)
	    	{
	    		ecrireLog("!! Erreur Comparaison DN : "+erreur) ;
	    	}	    
	    	return trouver ;
	}

	// Operation de maj	
	// Appelee via le tableau de valeur
	public void modifierLDAP(String dnLDAP, String[][] attributMajLDAP)
	{
	        if (contexteLDAP==null) return ;		
		// Modifier par exemple mon telephone et mon email
		// String[][] attributMajLDAP = {{"telephonenumber","010203040506"}, {"mail","lchentouf@valoris.com"}};
	        try
	        {
	        	// On liste l ensemble des operations elementaires 
	        	// 	maj        si DirContext.REPLACE_ATTRIBUTE
	        	// 	ajout      si DirContext.ADD_ATTRIBUTE 
	        	// 	suppresion si DirContext.REMOVE_ATTRIBUTE
	        	ModificationItem[] listeModificationLDAP = new ModificationItem[attributMajLDAP.length];
	   		for (int i=0;i<attributMajLDAP.length;i++)
          		{
        			Attribute modificationLDAP = new BasicAttribute(attributMajLDAP[i][0],attributMajLDAP[i][1]);
        			listeModificationLDAP[i]   = new ModificationItem(DirContext.REPLACE_ATTRIBUTE, modificationLDAP);  		
			}
			// Execution des modifications demandes
			contexteLDAP.modifyAttributes(dnLDAP, listeModificationLDAP);
			JOptionPane.showMessageDialog(this,"Mise a jour reussi sous "+dnLDAP,"Mise a jour",JOptionPane.INFORMATION_MESSAGE);
		}
	        catch (NamingException erreur)
	    	{
	    		ecrireLog("!! Erreur MAJ : "+erreur) ;
	    	}	    
	}
}
