# 
# ErrorHandler.pm 
# 
# This is an error handling class.  This class 
# handles runtime errors and maps error codes 
# to end user error message.  Messages can either 
# be returned to the caller, or the class can 
# issue a 302 redirect to the browser, sending 
# them to the designated error page. 
#
package ErrorHandler;

#
# import error codes
require ErrorCodes;

#
# import error messages
require ErrorMessages;

#
# We'll need to use CGI::escape()...
use CGI;

#
# constructor
#  Initializes data members
sub new {
   my $type = shift;
   my $self = {};

   #
   # Establish a default error page
   $self->{ErrorPage} = "/catalog/error.cgi";

   return bless $self, $type;
}

#
# Method: Error(ErrorCode, Message, Inline)
#
# This method takes an error code, an associated
# context message, and a flag indicating whether the
# end-user error message should be returned from the
# sub, or if a 302 redirect to $self->ErrorPage
# should be used instead.
#
sub Error {
	my ($self, $ErrorCode, $Message, $Inline) = @_;
	# Inline: 0 == return info, 1 == redirect

	#
	# Try to get object's context information
	#
	$CurrentPage = $ENV{"SCRIPT_NAME"};
	$RemoteIP = $ENV{"REMOTE_ADDR"};

	#
	# Do any xtra error logging here, or notify
	# support personal via pager/e-mail, or any other
	# type of activity-- writing to syslog, etc...
	#	

	#
	# If returning to the caller, call MapMessage
	# and return the string
	#
	if ($Inline == 0)
	{
		#
		# Return the result of MapMessage inline
		#
		return $self->MapMessage($ErrorCode);
	} else {
		#
		# URL Encode remote IP, Page Name,
		# and message
		#
		$encPage = CGI::escape($CurrentPage);
		$encRemip = CGI::escape($RemoteIP);
		$encMessage = CGI::escape($Message);

		#
		# Build redirection query string
		#
		$redirect = $self->{ErrorPage};
		$redirect = "$redirect?page=$encPage";
		$redirect = "$redirect&remip=$encRemip";
		$redirect = "$redirect&msg=$encMessage";
		$redirect = "$redirect&code=$ErrorCode";

		#
		# Do the redirect to the error page
		print("Location: $redirect\r\n");
		print("\r\n");
	}
}

#
# Method: MapMessage(ErrorCode)
# Returns the End-user error message associated
# with the error code.  NOTE: This mapping could
# be maintained in a separate file.
#
sub MapMessage {
	my ($self, $code) = @_;

	#
	# Map error code to message, and return
	# the message.
	#
	$retMessage = "No match.";

	if ( $code == $ErrorCodes::INCOMPLETE_URL)
	{
		$retMessage = $ErrorMessages::Messages[0];
	}

	if ( $code == $ErrorCodes::MISSING_FIELD )
	{
		$retMessage = $ErrorMessages::Messages[1];
	}

	if ( $code == $ErrorCodes::INCOMPLETE_POST)
	{
		$retMessage = $ErrorMessages::Messages[2];
	}
	
	if ( $code == $ErrorCodes::ILLEGAL_CHARACTERS)
	{
		$retMessage = $ErrorMessages::Messages[3];
	}

	return $retMessage;
}
1;
