package com.wrox.connectionpool2;

import java.sql.*;
import java.util.*;

/**
 * This class is a wrapper around a Connection, overriding the
 * close method to just inform the pool that it's available for
 * reuse again, and the isClosed method to return the state
 * of the wrapper instead of the Connection.
 */
class ConnectionWrapper implements Connection
{
   // realConn should be private but we use package scope to
   // be able to test removal of bad connections
   Connection realConn;
   private ConnectionPool pool;
   private boolean isClosed = false;

   public ConnectionWrapper(Connection realConn, ConnectionPool pool)
   {
      this.realConn = realConn;
      this.pool = pool;
   }

   /**
    * Inform the ConnectionPool that the ConnectionWrapper
    * is closed.
    */
   public void close() throws SQLException
   {
      isClosed = true;
      pool.wrapperClosed(realConn);
   }

   /**
    * Returns true if the ConnectionWrapper is closed, false
    * otherwise.
    */
   public boolean isClosed() throws SQLException
   {
      return isClosed;
   }

   /*
    * Wrapped methods.
    */
   public void clearWarnings() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.clearWarnings();
   }

   public void commit() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.commit();
   }

   public Statement createStatement() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.createStatement();
   }

   public boolean getAutoCommit() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.getAutoCommit();
   }

   public String getCatalog() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.getCatalog();
   }

   public DatabaseMetaData getMetaData() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.getMetaData();
   }

   public int getTransactionIsolation() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.getTransactionIsolation();
   }

   public SQLWarning getWarnings() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.getWarnings();
   }

   public boolean isReadOnly() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.isReadOnly();
   }

   public String nativeSQL(String sql) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.nativeSQL(sql);
   }

   public CallableStatement prepareCall(String sql) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.prepareCall(sql);
   }

   public PreparedStatement prepareStatement(String sql) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      return realConn.prepareStatement(sql);
   }

   public void rollback() throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.rollback();
   }

   public void setAutoCommit(boolean autoCommit) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.setAutoCommit(autoCommit);
   }

   public void setCatalog(String catalog) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.setCatalog(catalog);
   }

   public void setReadOnly(boolean readOnly) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.setReadOnly(readOnly);
   }

   public void setTransactionIsolation(int level) throws SQLException
   {
      if (isClosed)
      {
         throw new SQLException("Pooled connection is closed");
      }
      realConn.setTransactionIsolation(level);
   }
}
