//Source file: c:/work/wrox/case_study/code/BugTrackerApp/dataabstraction/PersistentDefect.java

package dataabstraction;

import businesslogic.*;
import businessmodel.*;
import apputil.*;
import java.sql.*;

/**
PersistentDefect is the foundation of the BugTracker application, and is the
class of object that is "remoted" through CORBA to the bugtracker client and 
servlet.

This class provides the object-relational / relational-object mapping; the clients 
of this class' objects need not know any details regarding the storage implementation.
*/
public class PersistentDefect extends Defect implements Persistent
{
	
	static Connection databaseConnection = null;
  static String driverName;
  static String sourceURL;
  static String userName;
  static String userPass;
  
  	
	public static final String QUERYBYKEY = "select defectid, product, versionreported, subsystem, defectsummary, defectdetail, state, priority, severity, reportedby, assignedto, datereported, datelastchanged, changedmaillist, resolvedmaillist, versionresolved, resolutionnotes, changedmodules, externalid from defects where defectid= ?";
  public static final String DROPBYKEY = "delete from defects where defectid = ?";
	public static final String INSERTDEFECT = "insert into defects (defectid, product, versionreported, subsystem, defectsummary, defectdetail, state, priority, severity, reportedby, assignedto, datereported, datelastchanged, changedmaillist, resolvedmaillist, versionresolved, resolutionnotes, changedmodules, externalid) values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
	public static final String UPDATEDEFECT = "update defects set product = ?, versionreported = ?, subsystem = ?, defectsummary = ?, defectdetail = ?, state = ?, priority = ?, severity = ?, reportedby = ?, assignedto = ?, datereported = ?, datelastchanged = ?, changedmaillist = ?, resolvedmaillist = ?, versionresolved = ?, resolutionnotes = ?, changedmodules = ?, externalid = ? where defectid = ?";
	/*
	this is one way to get the highest defect number, though not the most elegant.
	I've chosen this method rather than using the MAX function as this function 
	is not implemented in mSQL, one of the platforms that's being supported by 
	the BugTrackerApp.
	*/
	public static final String GETMAXDEFECTID = "select defectid from defects order by defectid desc";
	
	DefectPolicy m_DefectPolicy;
	
	/**
	This method ensures that the JDBC driver is loaded and
	we have a connection to the database. The first time it's
	called, it gets the driver name and datasource URL from the
	application properties and loads the database connection. 
	Subsequent calls to this method are essentially no-ops
	since the connection has already been established.
	*/
	
	protected void getDatabaseConnection() throws SQLException
	{
		
		if (databaseConnection == null)
		{
	
			AppConfig config = new AppConfig();
		
			try
			{
				driverName = config.getDatasourceDriver();
  			sourceURL = config.getDatasourceURL();
  			userName = config.getDatasourceUser();
  			userPass = config.getDatasourcePass();
	 			Class.forName (driverName);
  			databaseConnection = DriverManager.getConnection(sourceURL, userName, userPass);
			}
			catch(ClassNotFoundException cnfe)
			{
				throw new SQLException("Invalid driver specified in application properties");
			}
		}
 
	}
	
	
	/**
	Default constructor. Ensures that there is a connection to the data source.
	*/
	public PersistentDefect()
	{
		super();
	  try
		{
			getDatabaseConnection();
		}
		catch(SQLException sqle)
		{
			throw new RuntimeException("failed in PersistentDefect::PersistentDefect()\n" 
				+ sqle.toString());
		 }
		 
	 }
	
	/**
	Determines the highest number defect in database.
	@return maximum defect id
	*/
	public synchronized int getMaxDefectID() throws SQLException
	{
		
		int retval = 0;
		
		Statement query = databaseConnection.createStatement();
		
		/*
		selects defects by descending defect id. This is kind of a hack
		to work around the fact that mSQL doesn't support a MAX function,
		but this will work on pretty much any JDBC data source.	
		*/
		ResultSet rs = query.executeQuery(GETMAXDEFECTID);
		
		//any returns?
		if (rs.next())
		{
			retval = rs.getInt("defectid");
			rs.close();
		}
		
		return retval;
	}
	
	/**
	indicates whether the defect is stored in the database or not
	@return boolean value indicating whether the defect exists in the database or not
	*/
	public boolean rowExists()
	{
		
		boolean retval;
		
		PersistentDefect pd = new PersistentDefect();
		pd.setDefectID(getDefectID());
		try
		{
			pd.fetch();
			retval = true;
		}
		catch (SQLException sqle)
		{
			retval = false;
		}
		return retval;
		
	}
	
	/**
	Fetch the user object from the persistent store based on the user id
	@exception java.sql.SQLException 
	 */
	public void fetch() throws java.sql.SQLException 
	{
		int theDefectID = this.getDefectID();
		
		if (theDefectID == 0)
		{
			throw new SQLException("Cannot fetch defect without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(QUERYBYKEY);
		pQuery.setInt(1, theDefectID);
		
		ResultSet rs = pQuery.executeQuery();
		
		//any returns?
		if (rs.next())
		{
			String stringval;
			int intval;
			Date dateval;
			
			
			setDefectID(rs.getInt("defectid"));
			
			stringval = rs.getString("product");
			setProduct(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("versionreported");
			setVersionReported(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("subsystem");
			setSubsystem(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("defectsummary");
			setDefectSummary(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("defectdetail");
			setDefectDetails(rs.wasNull()?"":stringval);
			
			intval = rs.getInt("state");
			setState(rs.wasNull()?0:intval);
			
			intval = rs.getInt("priority");
			setPriority(rs.wasNull()?0:intval);
			
			intval = rs.getInt("severity");
			setSeverity(rs.wasNull()?0:intval);
			
			stringval = rs.getString("reportedby");
			setReportedBy(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("assignedto");
			setAssignedTo(rs.wasNull()?"":stringval);
			
			dateval = rs.getDate("datereported");
			setDateReported(rs.wasNull()?0:dateval.getTime());
			
			dateval = rs.getDate("datelastchanged");
			setDateLastChanged(rs.wasNull()?0:dateval.getTime());
			
			stringval = rs.getString("changedmaillist");
			setChangedMailList(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("resolvedmaillist");
			setResolvedMailList(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("versionresolved");
			setVersionResolved(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("resolutionnotes");
			setResolutionNotes(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("changedmodules");
			setChangedModules(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("externalid");
			setExternalId(rs.wasNull()?"":stringval);
		
			rs.close();
		}
		
		else
			throw new SQLException("No matching rows found!");
		
		}
	
	/**
	  Drops the defect data from the relational table(s)
	   */
	public void drop() throws java.sql.SQLException 
	{
		int theDefectID = this.getDefectID();
		
		if (theDefectID == 0)
		{
			throw new SQLException("Cannot drop defect without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(DROPBYKEY);
		pQuery.setInt(1, theDefectID);
		pQuery.executeUpdate();

		
		}
	
	/**
	   Inserts the defect data into the relational tables
	   
	   */
	public void insert() throws java.sql.SQLException 
	{
		
		int theDefectID = this.getDefectID();
		
		if (theDefectID == 0)
		{
			throw new SQLException("Cannot drop defect without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(INSERTDEFECT);
		
		pQuery.setInt(1, getDefectID());
		pQuery.setString(2, getProduct());
		pQuery.setString(3, getVersionReported());
		pQuery.setString(4, getSubsystem());
		pQuery.setString(5, getDefectSummary());
		pQuery.setString(6, getDefectDetails());
		pQuery.setInt(7, getState());
		pQuery.setInt(8, getPriority());
		pQuery.setInt(9, getSeverity());
		pQuery.setString(10, getReportedBy());
		pQuery.setString(11, getAssignedTo());

		/*
			a minor work around for a problem with the mSQL jdbc interface...
			need to pass the date as a string formatted *just* so...
			
			this will work just fine with other data sources, so doesn't
			too seriously compromise back-end independence.
		*/
    String theFormattedDate = apputil.util.LongToDateString(getDateReported());
		pQuery.setString(12, theFormattedDate);
    theFormattedDate = apputil.util.LongToDateString(getDateLastChanged());
		pQuery.setString(13, theFormattedDate);
		/* end workaround for mSQL */

		pQuery.setString(14, getChangedMailList());
		pQuery.setString(15, getResolvedMailList());
		pQuery.setString(16, getVersionResolved());
		pQuery.setString(17, getResolutionNotes());
		pQuery.setString(18, getChangedModules());
		pQuery.setString(19, getExternalId());

			
		pQuery.executeUpdate();
	
		
		}
	
	/**
	   Updates the defect information in the relational tables
	    */
	public void update() throws java.sql.SQLException 
	{
			int theDefectID = this.getDefectID();
		
		if (theDefectID == 0)
		{
			throw new SQLException("Cannot update defect without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(UPDATEDEFECT);

			
		pQuery.setString(1, getProduct());
		pQuery.setString(2, getVersionReported());
		pQuery.setString(3, getSubsystem());
		pQuery.setString(4, getDefectSummary());
		pQuery.setString(5, getDefectDetails());
		pQuery.setInt(6, getState());
		pQuery.setInt(7, getPriority());
		pQuery.setInt(8, getSeverity());
		pQuery.setString(9, getReportedBy());
		pQuery.setString(10, getAssignedTo());
		
		/*
			a minor work around for a problem with the mSQL jdbc interface...
			need to pass the date as a string formatted *just* so...
			
			this will work just fine with other data sources, so doesn't
			too seriously compromise back-end independence.
		*/
		String theFormattedDate = apputil.util.LongToDateString(getDateReported());
		pQuery.setString(11, theFormattedDate);
		theFormattedDate = apputil.util.LongToDateString(getDateLastChanged());
		pQuery.setString(12, theFormattedDate);
		/* end workaround for mSQL */
				
		pQuery.setString(13, getChangedMailList());
		pQuery.setString(14, getResolvedMailList());
		pQuery.setString(15, getVersionResolved());
		pQuery.setString(16, getResolutionNotes());
		pQuery.setString(17, getChangedModules());
		pQuery.setString(18, getExternalId());

		
		pQuery.setInt(19, getDefectID());
		
		pQuery.executeUpdate();
		
		}
		
		
	/**
	sets the policy object for this defect
	@param policy a DefectPolicy object
	*/	
	
	public void setPolicy(DefectPolicy policy)
	{
		m_DefectPolicy = policy;
	}	
		
	/**
	Attempt to store updates made to a defect
	@param userid the identity of the user requesting the change to be made
	@exception UpdateException thrown if the defect cannot be written
	@exception AppSecurityException thrown if the user does not have sufficient privileges for this operation
	*/	
	public void commit(String userid) throws UpdateException, AppSecurityException
	{
		
		super.commit(userid);

		m_DefectPolicy.securityCheck(this, userid);
		try
		{
			if (rowExists())
				update();
			else
				insert();
				
			// allow the policy object to process any notifications
			m_DefectPolicy.statusChanged(this);
				
		}
		catch(SQLException sqle)
		{
			String reason;
			reason = sqle.toString();
                        System.err.println(reason);
			throw new UpdateException();
		}
	};		
	
	/**
           
	 */
	public String toString() 
	{
		
			String retval = new String("PersistentDefect{" + 
			", Driver = " + driverName +
			", URL = " + sourceURL + 
			super.toString()) + "}";
			
			return retval;

		 
		}
		
 /**
	TestUnit implemented to support test harness
	*/
	
	public boolean TestUnit()
	{
		boolean retval;
		String newProduct = new String("BugTracker Turbo 2000");
		PersistentDefect test = new PersistentDefect();
				
		retval = super.TestUnit();
		if (retval)
		{
			retval = false;
			try
			{
				this.insert();
				test.setDefectID(this.getDefectID());
				test.fetch();
				
				if (test.getDefectID() == this.getDefectID())
					if(test.getState() == this.getState())
						if(test.getPriority() == this.getPriority())
							if(test.getSeverity() == this.getSeverity())
								if(apputil.util.LongToDateString(test.getDateLastChanged()).equals(apputil.util.LongToDateString(this.getDateLastChanged())))
									if(apputil.util.LongToDateString(test.getDateReported()).equals(apputil.util.LongToDateString(this.getDateReported())))
										if(test.getProduct().equals(this.getProduct()))
											if(test.getSubsystem().equals(this.getSubsystem()))
												if(test.getDefectSummary().equals(this.getDefectSummary()))
													if(test.getDefectDetails().equals(this.getDefectDetails()))
														if(test.getReportedBy().equals(this.getReportedBy()))
															if(test.getAssignedTo().equals(this.getAssignedTo()))
																if(test.getResolutionNotes().equals(this.getResolutionNotes()))
																	if(test.getChangedModules().equals(this.getChangedModules()))
																		if(test.getChangedMailList().equals(this.getChangedMailList()))
																			if(test.getResolvedMailList().equals(this.getResolvedMailList()))
																				if(test.getVersionResolved().equals(this.getVersionResolved()))
																					if(test.getExternalId().equals(this.getExternalId()))
																						retval = true;
				
				if (retval)
				{
					this.setProduct(newProduct);
					this.update();
					this.fetch();
					if (this.getProduct().equals(newProduct))
					{
						System.out.println("===>Manual test: check max id = " + this.getMaxDefectID());
						this.drop();
						try
						{
							this.fetch();
							retval = false;
						}
						catch (SQLException sqle)
						{
							// should have gotten here, since row is gone
							retval = true;
						}
					}
				}
			}		
			catch (SQLException sqle)
			{
				System.out.println(sqle.toString());
				retval = false;
			}
	
		}
			return retval;	
	}
		
}
