

/*

	file	:	UIDefectModel.java

*/

package ui;
import businessmodel.*;


import java.util.*;

/**
The UIDefectModel provides the observable model for the observing views 
of the user interface. This model coordinates updates to and from the model and the 
views that implement the DefectModelListenerInterface.

While this could have been implemented using the Observable class provided in the JDK,
this implementation is a bit cleaner and more tailored toward the application.
*/
public class UIDefectModel
{
	
	DefectIntf m_Defect;
	Vector m_Listeners;
		
		
	/** 
	default constructor
	*/	
	public UIDefectModel()
	{
		m_Defect = null;
		m_Listeners = new Vector();
	
	}
	
	/**
	Creates the model and uses the specified defect as the basis of the model
	@param defect the defect that provides the model basis
	*/
	public UIDefectModel(DefectIntf defect)
	{
		this.setActiveDefect(defect);
		m_Listeners = new Vector();
	}
	
	
	/** 
	Establishes the specified defect as the new basis for the model
	@param defect the defect that provides the model basis
	*/
	public void setActiveDefect(DefectIntf defect)
	{

		m_Defect = defect;
		UpdateListeners();

	}
	
	/**
	Determines if the logic model, including projections on the connected
	views, has been dirtied.
	@return true if the composite model is dirty, false if not
	*/
	public boolean isDirty()
	{

		boolean retval = false;
		
		if (m_Defect != null)
		{
			
			Iterator itor;
	
			itor = m_Listeners.iterator();
		
			while (itor.hasNext())
			{
				DefectModelListener aListener = 
					(DefectModelListener)itor.next();
					
				retval |= aListener.isDirty(m_Defect);
				
			}
		}
		
		return retval;
	
	}
	/**
	Accessor method
	*/
	public DefectIntf getActiveDefect()
	{
		
		return m_Defect;
		
	}
	
	/**
	Adds a new observer to the model
	@param listener new view to this model
	*/
	public synchronized void addDefectModelListener(DefectModelListener listener)
	{
		m_Listeners.addElement(listener);
		if (m_Defect != null)
		listener.ModelUpdated(m_Defect);
		
	}
	
	/**
	Causes the views to update themselves based on the contents of the model
	*/
	public void UpdateListeners()
	{
		Iterator itor;
				
		itor = m_Listeners.iterator();
		while (itor.hasNext())
		{
			DefectModelListener aListener = 
				(DefectModelListener)itor.next();
				
			aListener.ModelUpdated(m_Defect);
		}					
		
	}
	
	/**
	Causes the model to be updated from the information in the views
	*/
	public void UpdateModel()
	{
		Iterator itor;
				
		itor = m_Listeners.iterator();
		
		while (itor.hasNext())
		{
			DefectModelListener aListener = 
				(DefectModelListener)itor.next();
			if (aListener.isDirty(m_Defect))
				aListener.UpdateModel(m_Defect);
		}					
		
	}
	
}
	