import java.awt.*;
import java.io.*;
import java.util.*;
import java.net.URLEncoder;

import javax.servlet.*;
import javax.servlet.http.*;

import com.wrox.statistics.Stats;

public final class ChartServlet extends ImageServlet
{
  public void doGet(HttpServletRequest req, HttpServletResponse res)
      throws ServletException, IOException
  {
    res.setHeader("Pragma", "no-cache");
    res.setHeader("Cache-Control", "no-cache");
    res.setDateHeader("Expires", 0);

    Stats stats;
    if(req.getParameter("num") != null) stats = new Stats(req);
    else stats = new Stats(req.getPathInfo());

    boolean bw = Boolean.valueOf(req.getParameter("bw")).booleanValue();

    String img = req.getParameter("img");
    if(img != null) // Return an image
    {
      Color fg = Color.black, bg = new Color(0xB2B2B2), fill;
      if(bw) fill = new Color(0xFFFFFF);
      else fill = new Color(0xFF0000);

      if(img.equals("bar"))
	sendImage(res, createBarChart(200, stats, fg, bg, fill, bw));
      else if(img.equals("pie"))
	sendImage(res, createPieChart(200, 200, stats, fg, bg, fill, bw));
      else res.sendError(res.SC_BAD_REQUEST);
    }
    else // Return an HTML page
    {
      String statsString = stats.toString();
      res.setContentType("text/html");
      PrintWriter out = res.getWriter();

      out.println("<HTML><HEAD><TITLE>Statistics</TITLE></HEAD>");
      out.println("<BODY BGCOLOR=\"#B2B2B2\" TEXT=\"#000000\">");
      out.println("<H1>Statistics</H1>");
      out.println("<P><TABLE border=1 cellpadding=2>"+
		  "<TR><TH>Name</TH><TH>Count</TH><TH>%</TH></TR>");

      for(int i=0; i<stats.size(); i++)
	out.println("<TR><TD>"+stats.nameAt(i)+"</TD><TD>"+
		    stats.valueAt(i)+"</TD><TD>"+
		    stats.valueAt(i)*100/stats.getTotal()+"</TD></TR>");

      out.println("<TR><TD>Total</TD><TD>"+stats.getTotal()+
		  "</TD></TR></TABLE>");
      out.println("</P><P><IMG src=\""+req.getRequestURI()+
		  "?img=pie&bw="+bw+"&"+statsString+"\">");
      out.println("<IMG src=\""+req.getRequestURI()+
		  "?img=bar&bw="+bw+"&"+statsString+"\"></P>");

      if(bw)
      {
	out.println("<A href=\""+req.getRequestURI()+"?bw=false&"+
		    statsString+"\">[Color Version]</A>");
      }
      else
      {
	out.println("<A href=\""+req.getRequestURI()+"?bw=true&"+
		    statsString+"\">[Black & White Version]</A>");
      }
      out.println("<A href=\""+req.getRequestURI()+"?bw="+bw+
		  "\">[Update]</A>");

      out.println("</BODY></HTML>");

      out.close();
    }
  }


  private Image createPieChart(int w, int h, Stats stats,
			       Color fg, Color bg, Color fill,
			       boolean bw)
  {
    Image img = createImage(w, h, bg);
    Graphics g = img.getGraphics();

    // Draw segments
    {
      int done = 0, arc = 0;
      for(int i=0; i<stats.size(); i++)
      {
	done += stats.valueAt(i);
	int newArc = (done * 360) / stats.getTotal();
	g.setColor(fill);
	g.fillArc(0, 0, w-1, h-1, arc, newArc-arc);
	arc = newArc;
	if(bw) fill = rotateBrightness(fill);
	else fill = rotateColor(fill);
      }
    }

    // Draw border
    {
      g.setColor(fg);
      g.drawOval(0, 0, w-1, h-1);
      int done = 0;
      if(stats.size() > 1)
      {
	for(int i=0; i<stats.size(); i++)
	{
	  int arc = (done * 360) / stats.getTotal();
	  g.drawLine((w-1)/2, (h-1)/2,
		     (w-1)/2 + (int)(w * Math.cos((arc*Math.PI) / 180) / 2),
		     (h-1)/2 - (int)(h * Math.sin((arc*Math.PI) / 180) / 2));
	  done += stats.valueAt(i);
	}
      }
    }

    return img;
  }


  private Image createBarChart(int baseWidth, Stats stats,
			       Color fg, Color bg, Color fill,
			       boolean bw)
  {
    // Determine image size
    FontMetrics fm = getFontMetrics(getFont());
    int lh = fm.getHeight(), tw = 0, h = stats.size() * lh;
    for(int i=0; i<stats.size(); i++)
    {
      int lw = fm.stringWidth(stats.nameAt(i));
      if(lw > tw) tw = lw;
    }
    int w = tw + lh/2 + baseWidth;

    // Create image
    Image img = createImage(w, h, bg);
    Graphics g = img.getGraphics();

    // Draw bars and text
    int ascent = fm.getAscent(), descent = fm.getDescent();
    int y = fm.getMaxAscent();
    for(int i=0; i<stats.size(); i++)
    {
      g.setColor(fill);
      g.fillRect(w-baseWidth, y-ascent,
		 ((baseWidth*stats.valueAt(i))/stats.getMax())-1,
		 ascent+descent-1);
      g.setColor(fg);
      g.drawString(stats.nameAt(i),
		   tw - fm.stringWidth(stats.nameAt(i)), y);
      g.drawRect(w-baseWidth, y-ascent,
		 ((baseWidth*stats.valueAt(i))/stats.getMax())-1,
		 ascent+descent-1);
      y += lh;
      if(bw) fill = rotateBrightness(fill);
      else fill = rotateColor(fill);
    }

    return img;
  }


  private Color rotateColor(Color c)
  {
    float[] hsb = Color.RGBtoHSB(c.getRed(), c.getGreen(), c.getBlue(), null);
    hsb[0] += 0.095;
    if(hsb[0] > 1.0) hsb[0] -= 1.0;
    return new Color(Color.HSBtoRGB(hsb[0], hsb[1], hsb[2]));
  }


  private Color rotateBrightness(Color c)
  {
    float[] hsb = Color.RGBtoHSB(c.getRed(), c.getGreen(), c.getBlue(), null);
    hsb[2] -= 0.085;
    if(hsb[2] > 1.0) hsb[2] -= 1.0;
    return new Color(Color.HSBtoRGB(hsb[0], hsb[1], hsb[2]));
  }
}
