package org.genealogie.metier.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.regexp.RE;
import org.genealogie.log.Log;
import org.genealogie.metier.modele.Acte;
import org.genealogie.metier.modele.Auth;
import org.genealogie.metier.modele.Chrono;
import org.genealogie.metier.modele.Commune;
import org.genealogie.metier.modele.CommuneActe;
import org.genealogie.metier.modele.CommuneDetail;
import org.genealogie.metier.modele.Departement;
import org.genealogie.metier.modele.Mariage;
import org.genealogie.metier.modele.Pays;
import org.genealogie.metier.modele.Personne;
import org.genealogie.metier.modele.Recherche;
import org.genealogie.metier.modele.TypesActe;

public class GenealogieDAO {

	private Connection connection;


	public GenealogieDAO(Connection connection) {
		this.connection=connection;
	}

	private Connection getConnection() throws SQLException {
		return connection;
	}

	public int getNombrePersonnes() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(personne_id) as nb");
			requete.append(" from personne");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("nb");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}

	public int getNombrePersonnesFamille(Vector familles) {
		if( familles==null || familles.size()==0 ) {
			return 0;
		}

		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(personne_id) as nb");
			requete.append(" from personne,liaison_famille");
			requete.append(" where personne.personne_id=liaison_famille.personne_id");
			if( familles.size()>0 ) {
				requete.append(" and famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(" )");
			}
			ps=conn.prepareStatement(requete.toString());
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer id=(Integer)familles.elementAt(cpt);
				ps.setInt(cpt+1,id.intValue());
			}
			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("nb");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}

	public Personne getPere(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Personne pere=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select p1.personne_id,p1.personne_nom,p1.personne_prenom1,");
			requete.append("p1.personne_prenom2,p1.personne_prenom3");
			requete.append(" from personne p1,personne p2");
			requete.append(" where p1.personne_id=p2.personne_pere_id");
			requete.append(" and p2.personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				StringBuffer prenom=new StringBuffer();
				prenom.append(rs.getString("personne_prenom1"));
				if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
					prenom.append(" "+rs.getString("personne_prenom2"));
				if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
					prenom.append(" "+rs.getString("personne_prenom3"));
				pere=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return pere;
	}

	public Personne getMere(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Personne mere=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select p1.personne_id,p1.personne_nom,p1.personne_prenom1,");
			requete.append("p1.personne_prenom2,p1.personne_prenom3");
			requete.append(" from personne p1,personne p2");
			requete.append(" where p1.personne_id=p2.personne_mere_id");
			requete.append(" and p2.personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				StringBuffer prenom=new StringBuffer();
				prenom.append(rs.getString("personne_prenom1"));
				if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
					prenom.append(" "+rs.getString("personne_prenom2"));
				if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
					prenom.append(" "+rs.getString("personne_prenom3"));
				mere=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return mere;
	}

	public Personne getPersonne(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Personne personne=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select p1.personne_id,p1.personne_nom,p1.personne_prenom1,");
			requete.append("p1.personne_prenom2,p1.personne_prenom3");
			requete.append(" from personne p1");
			requete.append(" where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				StringBuffer prenom=new StringBuffer();
				prenom.append(rs.getString("personne_prenom1"));
				if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
					prenom.append(" "+rs.getString("personne_prenom2"));
				if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
					prenom.append(" "+rs.getString("personne_prenom3"));
				personne=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return personne;
	}
	

	public Chrono getChronologie(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne_date_naissance,personne_date_deces,");
			requete.append("personne_date_naissance_app,personne_date_deces_app,");
			requete.append("personne_commune_id_naissance,personne_commune_id_deces");
			requete.append(" from personne p");
			requete.append(" where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();

			SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
			if( rs.next() ) {
				String naissance="?";
				java.sql.Date dateNaissance=rs.getDate("personne_date_naissance");
				String dateNaissanceApp=rs.getString("personne_date_naissance_app");
				int lieuNaissanceId=rs.getInt("personne_commune_id_naissance");
				if( dateNaissance!=null )
					naissance=sdf.format(dateNaissance);
				else if( dateNaissanceApp!=null && !dateNaissanceApp.equals("") )
					naissance=dateNaissanceApp;
				if( lieuNaissanceId!=0 ) {
					Commune cu=getInfoCommune(lieuNaissanceId);
					naissance+=" - "+cu.getNom()+" ( "+cu.getDepartement()+")";
				}

				String deces="?";
				java.sql.Date dateDeces=rs.getDate("personne_date_deces");
				String dateDecesApp=rs.getString("personne_date_deces_app");
				int lieuDecesId=rs.getInt("personne_commune_id_deces");
				if( dateDeces!=null )
					deces=sdf.format(dateDeces);
				else if( dateDecesApp!=null && !dateDecesApp.equals("") )
					deces=dateDecesApp;
				if( lieuDecesId!=0 ) {
					Commune cu=getInfoCommune(lieuDecesId);
					deces+=" - "+cu.getNom()+" ( "+cu.getDepartement()+")";
				}

				Chrono cu=new Chrono(naissance,null,deces);
				return cu;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return null;
	}

	public Chrono getChronologieSimple(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne_date_naissance,personne_date_deces,");
			requete.append("personne_date_naissance_app,personne_date_deces_app");
			requete.append(" from personne p");
			requete.append(" where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();

			SimpleDateFormat sdf=new SimpleDateFormat("yyyy");
			if( rs.next() ) {
				String naissance="?";
				java.sql.Date dateNaissance=rs.getDate("personne_date_naissance");
				String dateNaissanceApp=rs.getString("personne_date_naissance_app");
				if( dateNaissance!=null )
					naissance=sdf.format(dateNaissance);
				else if( dateNaissanceApp!=null && !dateNaissanceApp.equals("") )
					naissance=dateNaissanceApp;

				String deces="?";
				java.sql.Date dateDeces=rs.getDate("personne_date_deces");
				String dateDecesApp=rs.getString("personne_date_deces_app");
				if( dateDeces!=null )
					deces=sdf.format(dateDeces);
				else if( dateDecesApp!=null && !dateDecesApp.equals("") )
					deces=dateDecesApp;

				Chrono cu=new Chrono(naissance,null,deces);
				return cu;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return null;
	}


	public Chrono getChronologie(int personneId,int conjointId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			Chrono chu=null;
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne_date_naissance,personne_date_deces,");
			requete.append("personne_date_naissance_app,personne_date_deces_app,");
			requete.append("personne_commune_id_naissance,personne_commune_id_deces");
			requete.append(" from personne p");
			requete.append(" where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();

			SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
			if( rs.next() ) {
				String naissance="?";
				java.sql.Date dateNaissance=rs.getDate("personne_date_naissance");
				String dateNaissanceApp=rs.getString("personne_date_naissance_app");
				int lieuNaissanceId=rs.getInt("personne_commune_id_naissance");
				if( dateNaissance!=null )
					naissance=sdf.format(dateNaissance);
				else if( dateNaissanceApp!=null && !dateNaissanceApp.equals("") )
					naissance=dateNaissanceApp;
				if( lieuNaissanceId!=0 ) {
					Commune cu=getInfoCommune(lieuNaissanceId);
					naissance+=" - "+cu.getNom()+" ( "+cu.getDepartement()+")";
				}

				String deces="?";
				java.sql.Date dateDeces=rs.getDate("personne_date_deces");
				String dateDecesApp=rs.getString("personne_date_deces_app");
				int lieuDecesId=rs.getInt("personne_commune_id_deces");
				if( dateDeces!=null )
					deces=sdf.format(dateDeces);
				else if( dateDecesApp!=null && !dateDecesApp.equals("") )
					deces=dateDecesApp;
				if( lieuDecesId!=0 ) {
					Commune cu=getInfoCommune(lieuDecesId);
					deces+=" - "+cu.getNom()+" ( "+cu.getDepartement()+")";
				}

				chu=new Chrono(naissance,null,deces);
			}

			Vector mariages=null;
			if( isHomme(personneId) )
				mariages=getMariages(personneId,1);
			else
				mariages=getMariages(personneId,0);

			String mariage="?";
			for(int cpt=0;cpt<mariages.size();cpt++) {
				Mariage mu=(Mariage)mariages.elementAt(cpt);
				if( mu.getIdConjoint()==conjointId ) {
					if( mu.getDateMariage()!=null ) {
						mariage=sdf.format(mu.getDateMariage());
					} else if( mu.getDateMariageApp()!=null && !mu.getDateMariageApp().equals("") )
						mariage=mu.getDateMariageApp();
					if( mu.getLieuMariage()!=null && !mu.getLieuMariage().equals("") ) {
						mariage+=" - "+mu.getLieuMariage();
					}

					if( chu==null )
						chu=new Chrono(null,mariage,null);
					else
						chu.setMariage(mariage);
					break;
				}
			}

			return chu;

		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return null;
	}

	public String getPersonneCommentaires(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne_commentaires");
			requete.append(" from personne");
			requete.append(" where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				String commentaires=rs.getString("personne_commentaires");
				if( commentaires!=null && commentaires.equals("") )
					return null;
				return commentaires;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return null;
	}

	public boolean isEnfantNaturel(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne.personne_enfant_naturel from personne");
			requete.append(" where personne.personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				if( rs.getInt("personne_enfant_naturel")==0 )
					return false;
				else
					return true;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return false;
	}

	public boolean isHomme(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select personne.personne_homme from personne");
			requete.append(" where personne.personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				if( rs.getInt("personne_homme")==0 )
					return false;
				else
					return true;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return false;
	}


	public Vector getFreresSoeurs(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector enfants=new Vector();
		Hashtable hash=new Hashtable();
		
		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select distinct p.personne_id,p.personne_nom,p.personne_prenom1,p.personne_prenom2,p.personne_prenom3 from personne p,personne p1");
			requete.append(" where p.personne_pere_id=p1.personne_pere_id");
			requete.append(" and p1.personne_id=? and p.personne_id<>?");
			requete.append(" and p.personne_pere_id is not null and p1.personne_pere_id is not null");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			ps.setInt(2,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				StringBuffer prenom=new StringBuffer();
				prenom.append(rs.getString("personne_prenom1"));
				if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
					prenom.append(" "+rs.getString("personne_prenom2"));
				if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
					prenom.append(" "+rs.getString("personne_prenom3"));
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
				if( hash.get(""+p.getId())==null ) {
					hash.put(""+p.getId(),"");
					enfants.addElement(p);
				}
			}

			requete=new StringBuffer();
			requete.append("select distinct p.personne_id,p.personne_nom,p.personne_prenom1,p.personne_prenom2,p.personne_prenom3 from personne p,personne p1");
			requete.append(" where p.personne_mere_id=p1.personne_mere_id");
			requete.append(" and p1.personne_id=? and p.personne_id<>?");
			requete.append(" and p.personne_mere_id is not null and p1.personne_mere_id is not null");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			ps.setInt(2,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				StringBuffer prenom=new StringBuffer();
				prenom.append(rs.getString("personne_prenom1"));
				if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
					prenom.append(" "+rs.getString("personne_prenom2"));
				if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
					prenom.append(" "+rs.getString("personne_prenom3"));
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
				if( hash.get(""+p.getId())==null ) {
					hash.put(""+p.getId(),"");
					enfants.addElement(p);
				}
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return enfants;
	}

	public Vector getEnfants(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector enfants=new Vector();

		try {
			conn=getConnection();

			String requete="select personne_id,personne_nom,personne_prenom1,personne_prenom2,personne_prenom3,personne_date_naissance from personne where personne_pere_id=? or personne_mere_id=? order by personne_date_naissance asc";
			ps=conn.prepareStatement(requete);
			ps.setInt(1,personneId);
			ps.setInt(2,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				String prenom=rs.getString("personne_prenom1");
				if( rs.getString("personne_prenom2")!=null )
					prenom+=" "+rs.getString("personne_prenom2");
				if( rs.getString("personne_prenom3")!=null )
					prenom+=" "+rs.getString("personne_prenom3");
				prenom=prenom.trim();
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom);
				enfants.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return enfants;
	}

	public Vector getEnfants(int pereId,int mereId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector enfants=new Vector();

		try {
			conn=getConnection();

			String requete="select personne_id,personne_nom,personne_prenom1,personne_prenom2,personne_prenom3 from personne where personne_pere_id=? and personne_mere_id=?";
			ps=conn.prepareStatement(requete);
			ps.setInt(1,pereId);
			ps.setInt(2,mereId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				String prenom=rs.getString("personne_prenom1");
				if( rs.getString("personne_prenom2")!=null )
					prenom+=" "+rs.getString("personne_prenom2");
				if( rs.getString("personne_prenom3")!=null )
					prenom+=" "+rs.getString("personne_prenom3");
				prenom=prenom.trim();
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom);
				enfants.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return enfants;
	}


	public Vector getMariages(int personneId,int homme) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector mariages=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select mariage.mariage_date_civil,mariage.mariage_date_religieux,");
			requete.append("mariage.mariage_date_civil_app,mariage.mariage_date_religieux_app,");
			requete.append("mariage.mariage_commune_id_civil,mariage.mariage_paroisse_id_religieux");
			if( homme==1 )
				requete.append(",conjoint.personne_id,conjoint.personne_nom,conjoint.personne_prenom1,conjoint.personne_prenom2,conjoint.personne_prenom3");
			else
				requete.append(",conjoint.personne_id,conjoint.personne_nom,conjoint.personne_prenom1,conjoint.personne_prenom2,conjoint.personne_prenom3");
			requete.append(" from mariage mariage,personne conjoint");
			if( homme==1 )
				requete.append(" where conjoint.personne_id=mariage.mariage_femme_id and mariage_mari_id=?");
			else
				requete.append(" where conjoint.personne_id=mariage.mariage_mari_id and mariage_femme_id=?");
			requete.append(" order by mariage.mariage_date_civil,mariage.mariage_date_religieux");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				java.sql.Date d=rs.getDate("mariage_date_civil");
				java.sql.Date d1=rs.getDate("mariage_date_religieux");
				String da=rs.getString("mariage_date_civil_app");
				String d1a=rs.getString("mariage_date_religieux_app");
				int l=rs.getInt("mariage_commune_id_civil");
				int l1=rs.getInt("mariage_paroisse_id_religieux");
				String lieu="";

				if( l!=0 ) {
					Commune cu=getInfoCommune(l);
					if( cu!=null )
						lieu=cu.getNom()+" ( "+cu.getDepartement()+" )";
				} else if( l1!=0 ) {
					Commune cu=getInfoCommune(l1);
					if( cu!=null )
						lieu=cu.getNom()+" ( "+cu.getDepartement()+" )";
				}

				String prenom=rs.getString("personne_prenom1");
				if( rs.getString("personne_prenom2")!=null )
					prenom+=" "+rs.getString("personne_prenom2");
				if( rs.getString("personne_prenom3")!=null )
					prenom+=" "+rs.getString("personne_prenom3");
				prenom=prenom.trim();

				if( d!=null ) {
					Mariage p=new Mariage(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom,d,null,lieu);
					mariages.addElement(p);
				} else if( da!=null && !da.equals("") ) {
					Mariage p=new Mariage(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom,null,da,lieu);
					mariages.addElement(p);
				} else if( d1!=null ) {
					Mariage p=new Mariage(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom,d1,null,lieu);
					mariages.addElement(p);
				} else if( d1a!=null && !d1a.equals("") ) {
					Mariage p=new Mariage(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom,null,d1a,lieu);
					mariages.addElement(p);
				} else {
					Mariage p=new Mariage(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom,null,null,lieu);
					mariages.addElement(p);
				}
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return mariages;
	}

	public boolean isParrainMarraine(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			String requete="select personne_id from personne where personne_parrain_id=? or personne_marraine_id=?";
			ps=conn.prepareStatement(requete);
			ps.setInt(1,personneId);
			ps.setInt(2,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				return true;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return false;
	}

	public boolean isTemoin(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			String requete="select mariage_id from mariage where mariage_temoin1_id=? or mariage_temoin2_id=? or mariage_temoin3_id=? or mariage_temoin4_id=?";
			ps=conn.prepareStatement(requete);
			ps.setInt(1,personneId);
			ps.setInt(2,personneId);
			ps.setInt(3,personneId);
			ps.setInt(4,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				return true;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return false;
	}

	public Vector getNomsFamille() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

//			String requete="select distinct(personne_nom) from PERSONNE order by personne_nom asc";
			String requete="select distinct personne_nom from PERSONNE order by personne_nom asc";
			ps=conn.prepareStatement(requete);
			rs=ps.executeQuery();
			while( rs.next() ) {
				noms.addElement(rs.getString("personne_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}

	public Vector getNomsFamille(boolean homme) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

			String requete="";
			if( homme )
				requete="select personne_nom from PERSONNE where personne_homme=1 order by personne_nom asc";
			else
				requete="select personne_nom from PERSONNE where personne_homme=0 order by personne_nom asc";
			ps=conn.prepareStatement(requete);
			rs=ps.executeQuery();
			while( rs.next() ) {
				noms.addElement(rs.getString("personne_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}

	public Vector getNomsFamille(boolean homme,int id) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

			String requete="";
			if( homme )
				requete="select personne_nom from PERSONNE where personne_homme=1 and personne_id<>"+id+" order by personne_nom asc";
			else
				requete="select personne_nom from PERSONNE where personne_homme=0 and personne_id<>"+id+" order by personne_nom asc";
			ps=conn.prepareStatement(requete);
			rs=ps.executeQuery();
			while( rs.next() ) {
				noms.addElement(rs.getString("personne_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}

	public Vector getListePersonnes(String nomFamille) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

			String requete="select personne_id,personne_nom,personne_prenom1 from personne where personne_nom=? order by personne_prenom1 asc";
			ps=conn.prepareStatement(requete);
			ps.setString(1,nomFamille);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),rs.getString("personne_prenom1"));
				noms.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}

	public Vector getListePersonnes(String nomFamille,boolean homme) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			if( homme ) {
				requete.append("select personne_id,personne_nom,personne_prenom1 from personne");
				requete.append(" where personne_nom=? and personne_homme=1 order by personne_prenom1 asc");
			} else {
				requete.append("select personne_id,personne_nom,personne_prenom1 from personne");
				requete.append(" where personne_nom=? and personne_homme=0 order by personne_prenom1 asc");
			}
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,nomFamille);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),rs.getString("personne_prenom1"));
				noms.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}

	public Vector getListePersonnes(String nomFamille,boolean homme,int id) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector noms=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			if( homme ) {
				requete.append("select personne_id,personne_nom,personne_prenom1 from personne");
				requete.append(" where personne_nom=? and personne_homme=1 and personne_id<>? order by personne_prenom1 asc");
			} else {
				requete.append("select personne_id,personne_nom,personne_prenom1 from personne");
				requete.append(" where personne_nom=? and personne_homme=0 and personne_id<>? order by personne_prenom1 asc");
			}
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,nomFamille);
			ps.setInt(2,id);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),rs.getString("personne_prenom1"));
				noms.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}


	public Vector getListeCommunesFamille(Vector familles,int borneSup) {
		Hashtable communes=new Hashtable();
		Vector listeCommunes=new Vector();

		if( familles==null || familles.size()==0 ) {
			return listeCommunes;
		}

		java.sql.Date dateBorneSup=null;
		SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
		ParsePosition pos=new ParsePosition(0);
		if( borneSup>0 ) {
			java.util.Date dateBorneSupTmp=sdf.parse("01/01/"+borneSup, pos);
			dateBorneSup=new java.sql.Date(dateBorneSupTmp.getTime());
		}

		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			//Communes de naissance, baptme, dcs et inhumation
			requete.append("select personne_commune_id_naissance,personne_commune_id_bapteme,");
			requete.append("personne_date_naissance,personne_date_bapteme,");
			requete.append("personne_commune_id_deces,personne_commune_id_inhumation,");
			requete.append("personne_date_deces,personne_date_inhumation");
			requete.append(" from personne,liaison_famille");
			requete.append(" where personne.personne_id=liaison_famille.personne_id");
			if( familles.size()>0 ) {
				requete.append(" and famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(" )");
			}
			ps=conn.prepareStatement(requete.toString());
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer familleId=(Integer)familles.elementAt(cpt);
				ps.setInt(cpt+1,familleId.intValue());
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				int idNaissance=rs.getInt("personne_commune_id_naissance");
				java.sql.Date dateNaissance=rs.getDate("personne_date_naissance");
				if( borneSup>0 ) {
					if( idNaissance!=0 && communes.get(new Integer(idNaissance))==null
								&& dateNaissance!=null && dateNaissance.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idNaissance),"");
				} else {
					if( idNaissance!=0 && communes.get(new Integer(idNaissance))==null )
						communes.put(new Integer(idNaissance),"");
				}

				int idBapteme=rs.getInt("personne_commune_id_bapteme");
				java.sql.Date dateBapteme=rs.getDate("personne_date_bapteme");
				if( borneSup>0 ) {
					if( idBapteme!=0 && communes.get(new Integer(idBapteme))==null
								&& dateBapteme!=null && dateBapteme.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idBapteme),"");
				} else {
					if( idBapteme!=0 && communes.get(new Integer(idBapteme))==null )
						communes.put(new Integer(idBapteme),"");
				}

				int idDeces=rs.getInt("personne_commune_id_deces");
				java.sql.Date dateDeces=rs.getDate("personne_date_deces");
				if( borneSup>0 ) {
					if( idDeces!=0 && communes.get(new Integer(idDeces))==null
								&& dateDeces!=null && dateDeces.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idDeces),"");
				} else {
					if( idDeces!=0 && communes.get(new Integer(idDeces))==null )
						communes.put(new Integer(idDeces),"");
				}

				int idInhumation=rs.getInt("personne_commune_id_inhumation");
				java.sql.Date dateInhumation=rs.getDate("personne_date_inhumation");
				if( borneSup>0 ) {
					if( idInhumation!=0 && communes.get(new Integer(idInhumation))==null
								&& dateInhumation!=null && dateInhumation.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idInhumation),"");
				} else {
					if( idInhumation!=0 && communes.get(new Integer(idInhumation))==null )
						communes.put(new Integer(idInhumation),"");
				}
			}

			//Communes de mariage civil ( mari )
			requete=new StringBuffer();
			requete.append("select mariage_commune_id_civil,mariage_paroisse_id_religieux,");
			requete.append("mariage_date_civil,mariage_date_religieux");
			requete.append(" from mariage,liaison_famille lf");
			requete.append(" where mariage.mariage_mari_id=lf.personne_id");
			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(" )");
			}
			ps=conn.prepareStatement(requete.toString());
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer familleId=(Integer)familles.elementAt(cpt);
				ps.setInt(cpt+1,familleId.intValue());
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				int idCivil=rs.getInt("mariage_commune_id_civil");
				java.sql.Date dateCivil=rs.getDate("mariage_date_civil");
				if( borneSup>0 ) {
					if( idCivil!=0 && communes.get(new Integer(idCivil))==null
								&& dateCivil!=null && dateCivil.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idCivil),"");
				} else {
					if( idCivil!=0 && communes.get(new Integer(idCivil))==null )
						communes.put(new Integer(idCivil),"");
				}

				int idReligieux=rs.getInt("mariage_paroisse_id_religieux");
				java.sql.Date dateReligieux=rs.getDate("mariage_date_religieux");
				if( borneSup>0 ) {
					if( idReligieux!=0 && communes.get(new Integer(idReligieux))==null
								&& dateReligieux!=null && dateReligieux.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idReligieux),"");
				} else {
					if( idReligieux!=0 && communes.get(new Integer(idReligieux))==null )
						communes.put(new Integer(idReligieux),"");
				}
			}

			//Communes de mariage civil ( femme )
			requete=new StringBuffer();
			requete.append("select mariage_commune_id_civil,mariage_paroisse_id_religieux,");
			requete.append("mariage_date_civil,mariage_date_religieux");
			requete.append(" from mariage,liaison_famille lf");
			requete.append(" where mariage.mariage_femme_id=lf.personne_id");
			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(" )");
			}
			ps=conn.prepareStatement(requete.toString());
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer familleId=(Integer)familles.elementAt(cpt);
				ps.setInt(cpt+1,familleId.intValue());
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				int idCivil=rs.getInt("mariage_commune_id_civil");
				java.sql.Date dateCivil=rs.getDate("mariage_date_civil");
				if( borneSup>0 ) {
					if( idCivil!=0 && communes.get(new Integer(idCivil))==null
								&& dateCivil.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idCivil),"");
				} else {
					if( idCivil!=0 && communes.get(new Integer(idCivil))==null )
						communes.put(new Integer(idCivil),"");
				}

				int idReligieux=rs.getInt("mariage_paroisse_id_religieux");
				java.sql.Date dateReligieux=rs.getDate("mariage_date_religieux");
				if( borneSup>0 ) {
					if( idReligieux!=0 && communes.get(new Integer(idReligieux))==null
								&& dateReligieux.compareTo(dateBorneSup)<0 )
						communes.put(new Integer(idReligieux),"");
				} else {
					if( idReligieux!=0 && communes.get(new Integer(idReligieux))==null )
						communes.put(new Integer(idReligieux),"");
				}
			}

			for(Enumeration e=communes.keys();e.hasMoreElements();) {
				Integer id=(Integer)e.nextElement();
				listeCommunes.addElement(id);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return listeCommunes;
	}

	public int getIdCommune(String nomCommune,String numeroDepartement,String nomPays) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune.commune_id");
			requete.append(" from pays,departement,commune");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_nom=? and departement.departement_numero=? and pays.pays_nom=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,nomCommune);
			ps.setString(2,numeroDepartement);
			ps.setString(3,nomPays);
			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("commune_id");
			}
			rs.close();
			rs=null;
			ps.close();
			ps=null;

			requete=new StringBuffer();
			requete.append("select commune.commune_id");
			requete.append(" from pays,departement,commune");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_nom=? and departement.departement_nom=? and pays.pays_nom=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,nomCommune);
			ps.setString(2,numeroDepartement);
			ps.setString(3,nomPays);
			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("commune_id");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}

	public Vector getListePays() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector pays=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select pays_id,pays_nom");
			requete.append(" from pays");
			requete.append(" order by pays_nom asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				Pays p=new Pays(rs.getInt("pays_id"),rs.getString("pays_nom"));
				pays.addElement(p);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return pays;
	}

	public Vector getListeDepartementsPays(String pays) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector departements=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select departement_id,departement_nom,departement_numero,pays_nom");
			requete.append(" from departement,pays");
			requete.append(" where departement.departement_pays_id=pays.pays_id");
			requete.append(" and upper(pays.pays_nom) like ?");
			requete.append(" order by departement_nom asc");
			ps=conn.prepareStatement(requete.toString());
			if( pays.endsWith("%") ) {
				ps.setString(1,pays.toUpperCase());
			} else {
				ps.setString(1,pays.toUpperCase()+"%");
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				Departement d=new Departement(rs.getInt("departement_id"),rs.getString("departement_nom"),
							rs.getString("departement_numero"),rs.getString("pays_nom"));
				departements.addElement(d);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return departements;
	}

	public Vector getListeDepartementsPays(int paysId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector departements=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select departement_id,departement_nom,departement_numero,pays_nom");
			requete.append(" from departement,pays");
			requete.append(" where departement_pays_id=?");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" order by departement_nom asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,paysId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Departement d=new Departement(rs.getInt("departement_id"),rs.getString("departement_nom"),rs.getString("departement_numero"),
						rs.getString("pays_nom"));
				departements.addElement(d);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return departements;
	}

	public Vector getListeDepartements(String nom) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector departements=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select departement_id,departement_nom,departement_numero,pays_nom");
			requete.append(" from departement,pays");
			requete.append(" where departement_nom like ?");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" order by departement_nom asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,"%"+nom+"%");
			rs=ps.executeQuery();
			while( rs.next() ) {
				Departement d=new Departement(rs.getInt("departement_id"),rs.getString("departement_nom"),rs.getString("departement_numero"),
						rs.getString("pays_nom"));
				departements.addElement(d);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return departements;
	}

	public Vector getListeDepartements(String nom,String pays) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector departements=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select departement_id,departement_nom,departement_numero,pays_nom");
			requete.append(" from departement,pays");
			requete.append(" where pays_nom like ?");
			requete.append(" and departement_nom like ?");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" order by departement_nom asc");
			ps=conn.prepareStatement(requete.toString());
			
			ps.setString(1,"%"+pays+"%");
			ps.setString(2,"%"+nom+"%");
			rs=ps.executeQuery();
			while( rs.next() ) {
				Departement d=new Departement(rs.getInt("departement_id"),rs.getString("departement_nom"),rs.getString("departement_numero"),
						rs.getString("pays_nom"));
				departements.addElement(d);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return departements;
	}

	public Commune getInfoCommune(int communeId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Commune commune=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_id=?");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,communeId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				commune=new Commune(rs.getInt("commune_id"),rs.getString("commune_nom"),rs.getString("departement_numero"),rs.getString("pays_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return commune;
	}

	public CommuneDetail getDetailCommune(int communeId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		CommuneDetail commune=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,departement_id,departement_nom,");
			requete.append("departement_numero,pays_id,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_id=?");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,communeId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				commune=new CommuneDetail(rs.getInt("commune_id"),rs.getString("commune_nom"),
					rs.getInt("departement_id"),rs.getString("departement_nom"),rs.getString("departement_numero"),
					rs.getInt("pays_id"),rs.getString("pays_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return commune;
	}

	public Vector getListeCommunes(String nomCommune) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector communes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_nom like ?");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			if( nomCommune.endsWith("%") )
				ps.setString(1,nomCommune);
			else
				ps.setString(1,nomCommune+"%");
			rs=ps.executeQuery();
			while( rs.next() ) {
				Commune c=new Commune(rs.getInt("commune_id"),rs.getString("commune_nom"),rs.getString("departement_numero"),rs.getString("pays_nom"));
				communes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return communes;
	}
	

	public Vector getListeCommunes(String nomCommune,String departement) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector communes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and ( commune.commune_nom like ? or commune.commune_nom_equivalent like ? )");
			requete.append(" and ( upper(departement.departement_nom) like ? or upper(departement.departement_numero) like ? )");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			if( nomCommune.endsWith("%") ) {
				ps.setString(1,nomCommune);
				ps.setString(2,nomCommune);
			} else {
				ps.setString(1,nomCommune+"%");
				ps.setString(2,nomCommune+"%");
			}
			if( departement.endsWith("%") ) {
				ps.setString(3,departement);
				ps.setString(4,departement);
			} else {
				ps.setString(3,departement+"%");
				ps.setString(4,departement+"%");
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				String nom=rs.getString("commune_nom");
				String nomEquivalent=rs.getString("commune_nom_equivalent");
				if( nomEquivalent!=null && !nomEquivalent.equals("") )
					nom+=" / "+nomEquivalent;
				Commune c=new Commune(rs.getInt("commune_id"),nom,
							rs.getString("departement_numero"),rs.getString("pays_nom"));
				communes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return communes;
	}

	public Vector getListeCommunesDepartement(String departement) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector communes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and ( upper(departement.departement_nom) like ? or upper(departement.departement_numero) like ? )");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			if( departement.endsWith("%") ) {
				ps.setString(1,departement);
				ps.setString(2,departement);
			} else {
				ps.setString(1,departement+"%");
				ps.setString(2,departement+"%");
			}
			rs=ps.executeQuery();
			while( rs.next() ) {
				String nom=rs.getString("commune_nom");
				String nomEquivalent=rs.getString("commune_nom_equivalent");
				if( nomEquivalent!=null && !nomEquivalent.equals("") )
					nom+=" / "+nomEquivalent;
				Commune c=new Commune(rs.getInt("commune_id"),nom,
							rs.getString("departement_numero"),rs.getString("pays_nom"));
				communes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return communes;
	}

	public Vector getListeCommunes(int departementId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector communes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select commune_id,commune_nom,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and departement.departement_id=?");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,departementId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Commune c=new Commune(rs.getInt("commune_id"),rs.getString("commune_nom"),rs.getString("departement_numero"),
																											rs.getString("pays_nom"));
				communes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return communes;
	}

	public Vector getListeActes() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,commune c,type_acte ta");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				actes.addElement(a);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	public Vector getListeActesCommune() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_auteur,a.acte_source,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				actes.addElement(a);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	public Vector getListeActesCommune(String commune) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" and c.commune_nom like '"+commune+"%'");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				actes.addElement(a);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	public Acte getActe(int acteId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" and acte_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acteId);
			rs=ps.executeQuery();
			if( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				return a;
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return null;
	}

	public Vector getListeActes(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();
		Hashtable ids=new Hashtable();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_auteur,a.acte_source,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c,liaison_acte la");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" and a.acte_id=la.acte_id");
			requete.append(" and la.personne_id=?");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				if( ids.get(new Integer(a.getId()))==null ) {
					ids.put(new Integer(a.getId()),"");
					actes.addElement(a);
				}
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	public Vector getListeActesId(int acteId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select distinct a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" and a.acte_id=?");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acteId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				actes.addElement(a);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}
	

	public Vector getListeActesFiche(int mariId,int femmeId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();
		Hashtable ids=new Hashtable();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(a.acte_id),a.acte_libelle,a.acte_url_image,a.acte_texte,a.acte_type_id,a.acte_date");
			requete.append("select a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c,liaison_acte la");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			requete.append(" and a.acte_id=la.acte_id");
			if( mariId!=0 && femmeId!=0 )
				requete.append(" and (la.personne_id=? or la.personne_id=?)");
			else if( mariId!=0 && femmeId==0 )
				requete.append(" and la.personne_id=?");
			else if( mariId==0 && femmeId!=0 )
				requete.append(" and la.personne_id=?");
			else
				requete.append(" and la.personne_id=0");
			requete.append(" order by acte_type_id asc");
			ps=conn.prepareStatement(requete.toString());
			if( mariId!=0 && femmeId!=0 ) {
				ps.setInt(1,mariId);
				ps.setInt(2,femmeId);
			} else if( mariId!=0 && femmeId==0 )
				ps.setInt(1,mariId);
			else if( mariId==0 && femmeId!=0 )
				ps.setInt(1,femmeId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
					rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
					rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
					rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
				if( ids.get(new Integer(a.getId()))==null ) {
					ids.put(new Integer(a.getId()),"");
					actes.addElement(a);
				}
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	public Vector getTypesActe() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector typesActe=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select type_acte_id,type_acte_nom");
			requete.append(" from type_acte");
			requete.append(" order by type_acte_id asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				TypesActe ta=new TypesActe(rs.getInt("type_acte_id"),rs.getString("type_acte_nom"));
				typesActe.addElement(ta);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return typesActe;
	}

	public Acte insertActe(Acte acte) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			int id=0;
//			ps=conn.prepareStatement("select NEXTVAL('acte_id_seq') as id");
			ps=conn.prepareStatement("select max(acte_id) as id from acte");
			rs=ps.executeQuery();
			if( rs.next() ) {
				id=rs.getInt("id")+1;
			}

			StringBuffer requete=new StringBuffer();
			requete.append("insert into acte values(?,?,?,?,?,?,?,?)");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,id);
			acte.setId(id);
			ps.setString(2,acte.getNom());
			ps.setInt(3,acte.getTypeId());
			ps.setDate(4,acte.getDate());
			ps.setInt(5,acte.getLieuId());
			ps.setString(6,acte.getCommentaires());
			ps.setString(7,acte.getMargeCommentaires());
			ps.setString(8,acte.getUrl());
			int nb=ps.executeUpdate();
			if( nb==0 )
				return null;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return null;
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return acte;
	}

	public boolean deleteActe(Acte acte) {
		Connection conn=null;
		PreparedStatement ps=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("delete from acte where acte_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acte.getId());
			int nb=ps.executeUpdate();
			if( nb==0 )
				return false;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return false;
		} finally {
			try {
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return true;
	}

	public boolean updateActe(Acte acte) {
		Connection conn=null;
		PreparedStatement ps=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("update acte set acte_libelle=?");
			requete.append(",acte_type_id=?");
			requete.append(",acte_date=?");
			requete.append(",acte_lieu_id=?");
			requete.append(",acte_texte=?");
			requete.append(",acte_marge_texte=?");
			requete.append(",acte_url_image=?");
			requete.append(" where acte_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setString(1,acte.getNom());
			ps.setInt(2,acte.getTypeId());
			ps.setDate(3,acte.getDate());
			ps.setInt(4,acte.getLieuId());
			ps.setString(5,acte.getCommentaires());
			ps.setString(6,acte.getMargeCommentaires());
			ps.setString(7,acte.getUrl());
			ps.setInt(8,acte.getId());
			int nb=ps.executeUpdate();
			if( nb==0 )
				return false;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return false;
		} finally {
			try {
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return true;
	}

	public int associerActe(int acteId,int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select * from liaison_acte");
			requete.append(" where acte_id=? and personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acteId);
			ps.setInt(2,personneId);
			rs=ps.executeQuery();
			if( rs.next() )
				return 1;

			requete=new StringBuffer();
			requete.append("insert into liaison_acte (acte_id,personne_id) values(?,?)");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acteId);
			ps.setInt(2,personneId);
			int nb=ps.executeUpdate();
			if( nb==0 )
				return 0;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return 0;
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 2;
	}

	public boolean deassocierActe(int acteId,int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("delete from liaison_acte where acte_id=? and personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,acteId);
			ps.setInt(2,personneId);
			int nb=ps.executeUpdate();
			if( nb==0 )
				return false;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return false;
		} finally {
			try {
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return true;
	}

	public boolean isAssocierActes(int personneId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select * from liaison_acte where personne_id=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			rs=ps.executeQuery();
			if( rs.next() )
				return true;
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			return false;
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return false;
	}


	public Vector getPagesSupplementaires(int pereId,int mereId) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector pages=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
//			requete.append("select distinct(classe_nom),page_ordre");
			requete.append("select distinct classe_nom,page_ordre");
			requete.append(" from page_supplementaire");
			if( pereId!=0 && mereId!=0 )
				requete.append(" where personne_id=? or personne_id=?");
			else if( pereId!=0 && mereId==0 )
				requete.append(" where personne_id=?");
			else if( pereId==0 && mereId!=0 )
				requete.append(" where personne_id=?");
			else
				return pages;
			requete.append(" order by page_ordre asc");
			ps=conn.prepareStatement(requete.toString());
			if( pereId!=0 && mereId!=0 ) {
				ps.setInt(1,pereId);
				ps.setInt(2,mereId);
			} else if( pereId!=0 && mereId==0 )
				ps.setInt(1,pereId);
			else if( pereId==0 && mereId!=0 )
				ps.setInt(1,mereId);
			rs=ps.executeQuery();
			while( rs.next() ) {
				pages.addElement(rs.getString("classe_nom"));
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return pages;
	}

	public int ajouterPagesSupplementaires(int personneId,String nomClasse,int ordre) {
		Connection conn=null;
		PreparedStatement ps=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("insert into page_supplementaire (personne_id,classe_nom,page_ordre) values(?,?,?)");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			ps.setString(2,nomClasse);
			ps.setInt(3,ordre);
			int nb=ps.executeUpdate();
			if( nb==0 )
				return 0;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return 0;
		} finally {
			try {
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 1;
	}

	public int supprimerPagesSupplementaires(int personneId,String nomClasse) {
		Connection conn=null;
		PreparedStatement ps=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("delete from page_supplementaire where personne_id=? and classe_nom=?");
			ps=conn.prepareStatement(requete.toString());
			ps.setInt(1,personneId);
			ps.setString(2,nomClasse);
			int nb=ps.executeUpdate();
			if( nb==0 )
				return 0;
			conn.commit();
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
			try {
				conn.rollback();
			} catch(SQLException ex1) {}
			return 0;
		} finally {
			try {
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 1;
	}

	public int getNombreActes() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(a.acte_id) as nb");
			requete.append(" from acte a");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("nb");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}
	
	public int getNombreActes(Auth auth) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(a.acte_id) as nb");
			requete.append(" from acte a,liaison_acte la,liaison_famille lf");
			requete.append(" where la.acte_id=a.acte_id");
			requete.append(" and la.personne_id=lf.personne_id");

			Vector familles=auth.getFamilles();
			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(")");
			}
			if( auth.getBorneSup()>0 )
				requete.append(" and a.acte_date<=?");

			ps=conn.prepareStatement(requete.toString());
			int indice=1;
			if( familles.size()>0 ) {
				for(int cpt=0;cpt<familles.size();cpt++) {
					int id=((Integer)familles.elementAt(cpt)).intValue();
					ps.setInt(indice,id);
					indice++;
				}
			}
			if( auth.getBorneSup()>0 ) {
				SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
				ParsePosition pos=new ParsePosition(0);
				int borneSup=auth.getBorneSup();
				java.util.Date tmp=sdf.parse("01/01/"+borneSup, pos);
				java.sql.Date dateBorneSup=new java.sql.Date(tmp.getTime());
				ps.setDate(indice,dateBorneSup);
			}

			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("nb");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}


	public int getNombreActes(Vector familles,int borneSup) {
		if( familles==null || familles.size()==0 ) {
			return 0;
		}

		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(a.acte_id) as nb");
			requete.append(" from acte a,liaison_acte l,liaison_famille lf");
			requete.append(" where a.acte_id=l.acte_id");
			requete.append(" and l.personne_id=lf.personne_id");
			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(" )");
			}
			if( borneSup>0 )
				requete.append(" and a.acte_date<=?");
			ps=conn.prepareStatement(requete.toString());
			int indice=1;
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer id=(Integer)familles.elementAt(cpt);
				ps.setInt(indice,id.intValue());
				indice++;
			}
			if( borneSup>0 ) {
				SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
				ParsePosition pos=new ParsePosition(0);
				java.util.Date tmp=sdf.parse("01/01/"+borneSup, pos);
				java.sql.Date dateBorneSup=new java.sql.Date(tmp.getTime());
				ps.setDate(indice,dateBorneSup);
			}

			rs=ps.executeQuery();
			if( rs.next() ) {
				return rs.getInt("nb");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return 0;
	}

	public Vector getListeCommunesActes() {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector villes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(acte_id) as nb,commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays,acte");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_id=acte.acte_lieu_id");
			requete.append(" group by commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String nomCommune=rs.getString("commune_nom");
				String nomCommuneEquivalent=rs.getString("commune_nom_equivalent");
				if( nomCommuneEquivalent!=null && !nomCommuneEquivalent.equals("") )
					nomCommune+=" / "+nomCommuneEquivalent;
				CommuneActe c=new CommuneActe(rs.getInt("commune_id"),nomCommune,rs.getString("departement_numero"),rs.getString("pays_nom"),rs.getInt("nb"));
				villes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return villes;
	}
	

	public Vector getListeCommunesActes(Auth auth) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector villes=new Vector();

		try {
			conn=getConnection();

			StringBuffer requete=new StringBuffer();
			requete.append("select count(acte.acte_id) as nb,commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" from commune,departement,pays,acte,liaison_acte la,liaison_famille lf");
			requete.append(" where commune.commune_departement_id=departement.departement_id");
			requete.append(" and departement.departement_pays_id=pays.pays_id");
			requete.append(" and commune.commune_id=acte.acte_lieu_id");
			requete.append(" and la.acte_id=acte.acte_id");
			requete.append(" and la.personne_id=lf.personne_id");

			Vector familles=auth.getFamilles();
			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(")");
			}

			requete.append(" group by commune_id,commune_nom,commune_nom_equivalent,departement_nom,departement_numero,pays_nom");
			requete.append(" order by commune_nom asc");
			ps=conn.prepareStatement(requete.toString());

			int indice=1;
			if( familles.size()>0 ) {
				for(int cpt=0;cpt<familles.size();cpt++) {
					int id=((Integer)familles.elementAt(cpt)).intValue();
					ps.setInt(indice,id);
					indice++;
				}
			}

			rs=ps.executeQuery();
			while( rs.next() ) {
				String nomCommune=rs.getString("commune_nom");
				String nomCommuneEquivalent=rs.getString("commune_nom_equivalent");
				if( nomCommuneEquivalent!=null && !nomCommuneEquivalent.equals("") )
					nomCommune+=" / "+nomCommuneEquivalent;
				CommuneActe c=new CommuneActe(rs.getInt("commune_id"),nomCommune,rs.getString("departement_numero"),rs.getString("pays_nom"),rs.getInt("nb"));
				villes.addElement(c);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return villes;
	}

	private boolean contient(String chaine1,RE r) {
		chaine1=chaine1.toLowerCase();
		StringTokenizer st=new StringTokenizer(chaine1,"\n");
		while( st.hasMoreTokens() ) {
			String c=st.nextToken();
			if( r.match(c) )
				return true;
		}
		return false;
	}
	

	public Vector getResultatsRechercheActes(String rechercher,boolean marge,boolean corps,java.sql.Date dateDebut,java.sql.Date dateFin,String ville) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			RE r = new RE(rechercher);
			conn=getConnection();
			StringBuffer requete=new StringBuffer();
			requete.append("select a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");
			if( dateDebut!=null )
				requete.append(" and a.acte_date>=?");
			if( dateFin!=null )
				requete.append(" and a.acte_date<=?");
			if( ville!=null && !ville.equals("") )
				requete.append(" and c.commune_nom like '"+ville+"%'");
			requete.append(" order by a.acte_date,a.acte_libelle asc");
			ps=conn.prepareStatement(requete.toString());
			int cpt=1;
			if( dateDebut!=null ) {
				ps.setDate(cpt,dateDebut);
				cpt++;
			}
			if( dateFin!=null )
				ps.setDate(cpt,dateFin);
			rs=ps.executeQuery();
			while( rs.next() ) {
				if( rechercher.equals("") ) {
					Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
						rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
						rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
						rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
					actes.addElement(a);
				} else if( (corps && contient(rs.getString("acte_texte"),r))
							|| (marge && contient(rs.getString("acte_marge_texte"),r))
							|| contient(rs.getString("acte_libelle"),r) ) {
					Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
						rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
						rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
						rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
					actes.addElement(a);
				}
			}
		} catch(Exception ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}


	public Vector getResultatsRechercheActes(String rechercher,boolean marge,boolean corps,java.sql.Date dateDebut,
								java.sql.Date dateFin,String ville,Auth auth) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector actes=new Vector();

		try {
			RE r = new RE(rechercher);
			conn=getConnection();
			StringBuffer requete=new StringBuffer();
			requete.append("select a.acte_id,a.acte_libelle,a.acte_url_image,a.acte_source,a.acte_auteur,a.acte_texte,a.acte_marge_texte,a.acte_type_id,a.acte_date,a.acte_lieu_id,");
			requete.append("c.commune_nom,ta.type_acte_nom");
			requete.append(" from acte a,type_acte ta,commune c,liaison_acte la,liaison_famille lf");
			requete.append(" where a.acte_type_id=ta.type_acte_id");
			requete.append(" and la.acte_id=a.acte_id");
			requete.append(" and la.personne_id=lf.personne_id");
			requete.append(" and a.acte_lieu_id=c.commune_id");

			Vector familles=auth.getFamilles();

			if( dateDebut!=null )
				requete.append(" and a.acte_date>=?");
			if( dateFin!=null )
				requete.append(" and a.acte_date<=?");
			if( ville!=null && !ville.equals("") )
				requete.append(" and c.commune_nom like '"+ville+"%'");

			if( familles.size()>0 ) {
				requete.append(" and lf.famille_id in (");
				for(int cpt=0;cpt<familles.size();cpt++) {
					if( cpt!=0 )
						requete.append(",?");
					else
						requete.append("?");
				}
				requete.append(")");
			}

			requete.append(" order by a.acte_date,a.acte_libelle asc");
			ps=conn.prepareStatement(requete.toString());
			int indice=1;
			if( dateDebut!=null ) {
				ps.setDate(indice,dateDebut);
				indice++;
			}
			if( dateFin!=null ) {
				ps.setDate(indice,dateFin);
				indice++;
			}

			if( familles.size()>0 ) {
				for(int cpt=0;cpt<familles.size();cpt++) {
					int id=((Integer)familles.elementAt(cpt)).intValue();
					ps.setInt(indice,id);
					indice++;
				}
			}

			rs=ps.executeQuery();
			while( rs.next() ) {
				if( rechercher.equals("") ) {
					Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
						rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
						rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
						rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
					actes.addElement(a);
				} else if( (corps && contient(rs.getString("acte_texte"),r))
							|| (marge && contient(rs.getString("acte_marge_texte"),r))
							|| contient(rs.getString("acte_libelle"),r) ) {
					Acte a=new Acte(rs.getInt("acte_id"),rs.getString("acte_libelle"),
						rs.getString("acte_url_image"),rs.getDate("acte_date"),rs.getInt("acte_lieu_id"),rs.getString("commune_nom"),
						rs.getInt("acte_type_id"),rs.getString("type_acte_nom"),rs.getString("acte_source"),
						rs.getString("acte_auteur"),rs.getString("acte_texte"),rs.getString("acte_marge_texte"));
					actes.addElement(a);
				}
			}
		} catch(Exception ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return actes;
	}

	private Vector trierPersonnes(Vector personnes) {
		Vector newPersonnes=new Vector();
		for(int cpt=0;cpt<personnes.size();cpt++) {
			Personne personne=(Personne)personnes.elementAt(cpt);
			boolean insere=false;
			for(int cpt1=0;cpt1<newPersonnes.size();cpt1++) {
				Personne personne1=(Personne)newPersonnes.elementAt(cpt1);
				if( (personne.getNom()).compareTo(personne1.getNom())<0 ) {
					newPersonnes.insertElementAt(personne,cpt1);
					insere=true;
					break;
				}
			}
			if( !insere )
				newPersonnes.addElement(personne);
		}
		return newPersonnes;
	}

	public Vector getResultatsRecherche(Recherche rechercheUtils) {
		return getResultatsRecherche(rechercheUtils,null);
	}

	private StringBuffer construitRequeteRechercheReligieuxSF(Recherche rechercheUtils) {
		String condition;
		StringBuffer requete2=new StringBuffer();
		requete2.append("select personne1.personne_id as mari_id,personne1.personne_nom as mari_nom,");
		requete2.append("personne1.personne_prenom1 as mari_prenom,");
		requete2.append("personne2.personne_id as femme_id,personne2.personne_nom as femme_nom,");
		requete2.append("personne2.personne_prenom1 as femme_prenom");
		requete2.append(" from personne personne1,personne personne2,mariage mariage");

		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(",commune commune1");
		}

		requete2.append(" where personne1.personne_id=mariage.mariage_mari_id and personne2.personne_id=mariage.mariage_femme_id");
		condition="and";
		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_paroisse_id_religieux=commune1.commune_id");
		}

		if( !rechercheUtils.getNom().equals("") ) {
			requete2.append(" "+condition+" ( LOWER(personne1.personne_nom)='"+rechercheUtils.getNom()+"'");
			requete2.append(" or LOWER(personne2.personne_nom)='"+rechercheUtils.getNom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete2.append(" "+condition+" ( LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			requete2.append(" or LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(" "+condition+" commune1.commune_nom like '%"+rechercheUtils.getVille()+"%'");
		}

		if( !rechercheUtils.getDateDebut().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_date_religieux>=?");
			condition="and";
		}

		if( !rechercheUtils.getDateFin().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_date_religieux<=?");
			condition="and";
		}
		return requete2;
	}

	private StringBuffer construitRequeteRechercheCivilSF(Recherche rechercheUtils) {
		StringBuffer requete1=new StringBuffer();
		requete1.append("select personne1.personne_id as mari_id,personne1.personne_nom as mari_nom,");
		requete1.append("personne1.personne_prenom1 as mari_prenom,");
		requete1.append("personne2.personne_id as femme_id,personne2.personne_nom as femme_nom,");
		requete1.append("personne2.personne_prenom1 as femme_prenom");
		requete1.append(" from personne personne1,personne personne2,mariage mariage");

		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(",commune commune1");
		}

		requete1.append(" where personne1.personne_id=mariage.mariage_mari_id and personne2.personne_id=mariage.mariage_femme_id");
		String condition="and";
		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_commune_id_civil=commune1.commune_id");
		}

		if( !rechercheUtils.getNom().equals("") ) {
			requete1.append(" "+condition+" ( LOWER(personne1.personne_nom)='"+rechercheUtils.getNom()+"'");
			requete1.append(" or LOWER(personne2.personne_nom)='"+rechercheUtils.getNom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete1.append(" "+condition+" ( LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			requete1.append(" or LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(" "+condition+" commune1.commune_nom like '%"+rechercheUtils.getVille()+"%'");
		}

		if( !rechercheUtils.getDateDebut().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_date_civil>=?");
			condition="and";
		}

		if( !rechercheUtils.getDateFin().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_date_civil<=?");
			condition="and";
		}
		return requete1;
	}

	private StringBuffer construitRequeteRechercheSF(Recherche rechercheUtils) {
		StringBuffer requete=new StringBuffer();
		requete.append("select personne.personne_id,personne.personne_nom,personne.personne_prenom1,personne.personne_prenom2,personne.personne_prenom3");
		requete.append(" from personne personne");
		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() )
				requete.append(",commune commune1");
			if( rechercheUtils.getBapteme() )
				requete.append(",commune commune2");
			if( rechercheUtils.getDeces() )
				requete.append(",commune commune3");
			if( rechercheUtils.getInhumation() )
				requete.append(",commune commune4");
		}
		String condition="where";

		if( !rechercheUtils.getNom().equals("") ) {
			requete.append(" "+condition+" LOWER(personne.personne_nom)='"+rechercheUtils.getNom()+"'");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete.append(" "+condition+" LOWER(personne.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_commune_id_naissance=commune1.commune_id");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_commune_id_bapteme=commune2.commune_id");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_commune_id_deces=commune3.commune_id");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_commune_id_inhumation=commune4.commune_id");
				condition="and";
			}
		}

		//Lieux
		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" commune1.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" commune2.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" commune3.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" commune4.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
		}

		//Dates
		if( !rechercheUtils.getDateDebut().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_date_naissance>=?");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_date_bapteme>=?");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_date_deces>=?");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_date_inhumation>=?");
				condition="and";
			}
		}
		if( !rechercheUtils.getDateFin().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_date_naissance<=?");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_date_bapteme<=?");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_date_deces<=?");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_date_inhumation<=?");
				condition="and";
			}
		}

		requete.append(" order by personne_nom,personne_prenom1");
		return requete;
	}

	public Vector getResultatsRecherche(Recherche rechercheUtils,Vector familles) {
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;
		Vector personnes=new Vector();
		Hashtable cles=new Hashtable();

		try {
			conn=getConnection();

			if( !rechercheUtils.getMariage() ) {
				StringBuffer requete;
				if (familles==null) {
					requete = construitRequeteRechercheSF(rechercheUtils);
				} else {
					requete = construitRequeteRecherche(rechercheUtils, familles);
				}

				
				ps=conn.prepareStatement(requete.toString());

				int cpt=1;
				if( !rechercheUtils.getDateDebut().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateDebut(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());

					if( rechercheUtils.getNaissance() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getBapteme() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getDeces() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getInhumation() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
				}
				if( !rechercheUtils.getDateFin().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateFin(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());

					if( rechercheUtils.getNaissance() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getBapteme() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getDeces() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
					if( rechercheUtils.getInhumation() ) {
						ps.setDate(cpt,d1);
						cpt++;
					}
				}

				rs=ps.executeQuery();
				while( rs.next() ) {
					StringBuffer prenom=new StringBuffer();
					prenom.append(rs.getString("personne_prenom1"));
					if( rs.getString("personne_prenom2")!=null && !rs.getString("personne_prenom2").equals("") )
						prenom.append(" "+rs.getString("personne_prenom2"));
					if( rs.getString("personne_prenom3")!=null && !rs.getString("personne_prenom3").equals("") )
						prenom.append(" "+rs.getString("personne_prenom3"));
					Personne p=new Personne(rs.getInt("personne_id"),rs.getString("personne_nom"),prenom.toString());
					cles.put(new Integer(p.getId()),"");
					personnes.addElement(p);
				}
			}

			if( rechercheUtils.getMariage() ) {
				StringBuffer requete1;
				if (familles==null) {
					requete1 = construitRequeteRechercheCivilSF(rechercheUtils);
				} else {
					requete1 = construitRequeteRechercheCivil(rechercheUtils, familles);
				}

				Log.trace("requete1 = "+requete1.toString());
				ps=conn.prepareStatement(requete1.toString());

				int cpt=1;
				if( !rechercheUtils.getDateDebut().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateDebut(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());
					ps.setDate(cpt,d1);
					cpt++;
				}
				if( !rechercheUtils.getDateFin().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateFin(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());
					ps.setDate(cpt,d1);
					cpt++;
				}

				rs=ps.executeQuery();
				construitResultatRecherche(rechercheUtils, rs, personnes, cles);

				StringBuffer requete2;
				if (familles==null) {
					requete2 = construitRequeteRechercheReligieuxSF(rechercheUtils);
				} else {
					requete2 = construitRequeteRechercheReligieux(rechercheUtils, familles);
				}

				Log.trace("requete2 = "+requete2.toString());
				ps=conn.prepareStatement(requete2.toString());

				cpt=1;
				if( !rechercheUtils.getDateDebut().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateDebut(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());
					ps.setDate(cpt,d1);
					cpt++;
				}
				if( !rechercheUtils.getDateFin().equals("") ) {
					SimpleDateFormat sdf=new SimpleDateFormat("dd/MM/yyyy");
					ParsePosition pos=new ParsePosition(0);
					java.util.Date d=sdf.parse(rechercheUtils.getDateFin(), pos);
					java.sql.Date d1=new java.sql.Date(d.getTime());
					ps.setDate(cpt,d1);
					cpt++;
				}

				rs=ps.executeQuery();
				construitResultatRecherche(rechercheUtils, rs, personnes, cles);
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return personnes;
	}


	private void construitResultatRecherche(Recherche rechercheUtils, ResultSet rs, Vector personnes, Hashtable cles) throws SQLException {
		while( rs.next() ) {
			Personne mari=new Personne(rs.getInt("mari_id"),rs.getString("mari_nom"),rs.getString("mari_prenom"));
			Personne femme=new Personne(rs.getInt("femme_id"),rs.getString("femme_nom"),rs.getString("femme_prenom"));
	
			String mariNom=mari.getNom().toLowerCase();
			String mariPrenom=mari.getPrenom().toLowerCase();
			String femmeNom=femme.getNom().toLowerCase();
			String femmePrenom=femme.getPrenom().toLowerCase();
	
			if( cles.get(new Integer(mari.getId()))==null ) {
				construitResultatRechercheMF(rechercheUtils, personnes, cles, mari, mariNom, mariPrenom);
			}
			if( cles.get(new Integer(femme.getId()))==null ) {
				construitResultatRechercheMF(rechercheUtils, personnes, cles, femme, femmeNom, femmePrenom);
			}
		}
	}

	private void construitResultatRechercheMF(Recherche rechercheUtils, Vector personnes, Hashtable cles, Personne femme, String femmeNom, String femmePrenom) {
		if( !rechercheUtils.getNom().equals("") && !rechercheUtils.getPrenom().equals("") ) {
			if( femmeNom.equals(rechercheUtils.getNom()) && femmePrenom.equals(rechercheUtils.getPrenom()) ) {
				personnes.addElement(femme);
				cles.put(new Integer(femme.getId()),"");
			}
		} else if( !rechercheUtils.getNom().equals("") && rechercheUtils.getPrenom().equals("") ) {
			if( femmeNom.equals(rechercheUtils.getNom()) ) {
				personnes.addElement(femme);
				cles.put(new Integer(femme.getId()),"");
			}
		} else if( rechercheUtils.getNom().equals("") && !rechercheUtils.getPrenom().equals("") ) {
			if( femmePrenom.equals(rechercheUtils.getPrenom()) ) {
				personnes.addElement(femme);
				cles.put(new Integer(femme.getId()),"");
			}
		} else if( rechercheUtils.getNom().equals("") && rechercheUtils.getPrenom().equals("") ) {
			personnes.addElement(femme);
			cles.put(new Integer(femme.getId()),"");
		}
	}

	private StringBuffer construitRequeteRechercheReligieux(Recherche rechercheUtils, Vector familles) {
		String condition;
		StringBuffer requete2=new StringBuffer();
		requete2.append("select personne1.personne_id as mari_id,personne1.personne_nom as mari_nom,");
		requete2.append("personne1.personne_prenom1 as mari_prenom,");
		requete2.append("personne2.personne_id as femme_id,personne2.personne_nom as femme_nom,");
		requete2.append("personne2.personne_prenom1 as femme_prenom");
		requete2.append(" from personne personne1,personne personne2,mariage mariage,liaison_famille lf1,liaison_famille lf2");
		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(",commune commune1");
		}
		requete2.append(" where personne1.personne_id=lf1.personne_id");
		requete2.append(" and personne2.personne_id=lf2.personne_id");
		if( familles.size()>0 ) {
			requete2.append(" and ( lf1.famille_id in ( ");
			for(int cpt2=0;cpt2<familles.size();cpt2++) {
				Integer id=(Integer)familles.elementAt(cpt2);
				if( cpt2!=0 )
					requete2.append(","+id.intValue());
				else
					requete2.append(""+id.intValue());
			}
			requete2.append(" )");

			requete2.append(" or lf2.famille_id in ( ");
			for(int cpt2=0;cpt2<familles.size();cpt2++) {
				Integer id=(Integer)familles.elementAt(cpt2);
				if( cpt2!=0 )
					requete2.append(","+id.intValue());
				else
					requete2.append(""+id.intValue());
			}
			requete2.append(" ) )");
		}

		requete2.append(" and personne1.personne_id=mariage.mariage_mari_id and personne2.personne_id=mariage.mariage_femme_id");
		condition="and";
		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_paroisse_id_religieux=commune1.commune_id");
		}

		if( !rechercheUtils.getNom().equals("") ) {
			requete2.append(" "+condition+" ( LOWER(personne1.personne_nom)='"+rechercheUtils.getNom()+"'");
			requete2.append(" or LOWER(personne2.personne_nom)='"+rechercheUtils.getNom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete2.append(" "+condition+" ( LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			requete2.append(" or LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			requete2.append(" "+condition+" commune1.commune_nom like '%"+rechercheUtils.getVille()+"%'");
		}

		if( !rechercheUtils.getDateDebut().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_date_religieux>=?");
			condition="and";
		}

		if( !rechercheUtils.getDateFin().equals("") ) {
			requete2.append(" "+condition+" mariage.mariage_date_religieux<=?");
			condition="and";
		}
		return requete2;
	}

	private StringBuffer construitRequeteRechercheCivil(Recherche rechercheUtils, Vector familles) {
		StringBuffer requete1=new StringBuffer();
		requete1.append("select personne1.personne_id as mari_id,personne1.personne_nom as mari_nom,");
		requete1.append("personne1.personne_prenom1 as mari_prenom,");
		requete1.append("personne2.personne_id as femme_id,personne2.personne_nom as femme_nom,");
		requete1.append("personne2.personne_prenom1 as femme_prenom");
		requete1.append(" from personne personne1,personne personne2,mariage mariage,liaison_famille lf1,liaison_famille lf2");
		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(",commune commune1");
		}
		requete1.append(" where personne1.personne_id=lf1.personne_id");
		requete1.append(" and personne2.personne_id=lf2.personne_id");
		if( familles.size()>0 ) {
			requete1.append(" and ( lf1.famille_id in ( ");
			for(int cpt1=0;cpt1<familles.size();cpt1++) {
				Integer id=(Integer)familles.elementAt(cpt1);
				if( cpt1!=0 )
					requete1.append(","+id.intValue());
				else
					requete1.append(""+id.intValue());
			}
			requete1.append(" )");

			requete1.append(" or lf2.famille_id in ( ");
			for(int cpt1=0;cpt1<familles.size();cpt1++) {
				Integer id=(Integer)familles.elementAt(cpt1);
				if( cpt1!=0 )
					requete1.append(","+id.intValue());
				else
					requete1.append(""+id.intValue());
			}
			requete1.append(" ) )");
		}

		requete1.append(" and personne1.personne_id=mariage.mariage_mari_id and personne2.personne_id=mariage.mariage_femme_id");
		String condition="and";
		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_commune_id_civil=commune1.commune_id");
		}

		if( !rechercheUtils.getNom().equals("") ) {
			requete1.append(" "+condition+" ( LOWER(personne1.personne_nom)='"+rechercheUtils.getNom()+"'");
			requete1.append(" or LOWER(personne2.personne_nom)='"+rechercheUtils.getNom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete1.append(" "+condition+" ( LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			requete1.append(" or LOWER(personne2.personne_prenom1)='"+rechercheUtils.getPrenom()+"' )");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			requete1.append(" "+condition+" commune1.commune_nom like '%"+rechercheUtils.getVille()+"%'");
		}

		if( !rechercheUtils.getDateDebut().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_date_civil>=?");
			condition="and";
		}

		if( !rechercheUtils.getDateFin().equals("") ) {
			requete1.append(" "+condition+" mariage.mariage_date_civil<=?");
			condition="and";
		}
		return requete1;
	}

	private StringBuffer construitRequeteRecherche(Recherche rechercheUtils, Vector familles) {
		StringBuffer requete=new StringBuffer();
		requete.append("select personne.personne_id,personne.personne_nom,personne.personne_prenom1,personne.personne_prenom2,personne.personne_prenom3");
		requete.append(" from personne personne,liaison_famille lf");
		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() )
				requete.append(",commune commune1");
			if( rechercheUtils.getBapteme() )
				requete.append(",commune commune2");
			if( rechercheUtils.getDeces() )
				requete.append(",commune commune3");
			if( rechercheUtils.getInhumation() )
				requete.append(",commune commune4");
		}
		requete.append(" where personne.personne_id=lf.personne_id");
		if( familles.size()>0 ) {
			requete.append(" and lf.famille_id in ( ");
			for(int cpt=0;cpt<familles.size();cpt++) {
				Integer id=(Integer)familles.elementAt(cpt);
				if( cpt!=0 )
					requete.append(","+id.intValue());
				else
					requete.append(""+id.intValue());
			}
			requete.append(" )");
		}
		
		String condition="and";

		if( !rechercheUtils.getNom().equals("") ) {
			requete.append(" "+condition+" LOWER(personne.personne_nom)='"+rechercheUtils.getNom()+"'");
			condition="and";
		}

		if( !rechercheUtils.getPrenom().equals("") ) {
			requete.append(" "+condition+" LOWER(personne.personne_prenom1)='"+rechercheUtils.getPrenom()+"'");
			condition="and";
		}

		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_commune_id_naissance=commune1.commune_id");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_commune_id_bapteme=commune2.commune_id");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_commune_id_deces=commune3.commune_id");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_commune_id_inhumation=commune4.commune_id");
				condition="and";
			}
		}

		//Lieux
		if( !rechercheUtils.getVille().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" commune1.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" commune2.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" commune3.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" commune4.commune_nom='"+rechercheUtils.getVille()+"'");
				condition="and";
			}
		}

		//Dates
		if( !rechercheUtils.getDateDebut().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_date_naissance>=?");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_date_bapteme>=?");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_date_deces>=?");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_date_inhumation>=?");
				condition="and";
			}
		}
		if( !rechercheUtils.getDateFin().equals("") ) {
			if( rechercheUtils.getNaissance() ) {
				requete.append(" "+condition+" personne.personne_date_naissance<=?");
				condition="and";
			}
			if( rechercheUtils.getBapteme() ) {
				requete.append(" "+condition+" personne.personne_date_bapteme<=?");
				condition="and";
			}
			if( rechercheUtils.getDeces() ) {
				requete.append(" "+condition+" personne.personne_date_deces<=?");
				condition="and";
			}
			if( rechercheUtils.getInhumation() ) {
				requete.append(" "+condition+" personne.personne_date_inhumation<=?");
				condition="and";
			}
		}
		requete.append(" order by personne_nom,personne_prenom1");
		return requete;
	}

	public Hashtable getNomsCommunes() {
		Hashtable noms=new Hashtable();
		Connection conn=null;
		PreparedStatement ps=null;
		ResultSet rs=null;

		try {
			conn=getConnection();

			//Check naissance
			StringBuffer requete=new StringBuffer();
			requete.append("select personne.personne_nom,commune.commune_nom,departement.departement_numero");
			requete.append(" from personne personne,commune commune, departement departement");
			requete.append(" where personne.personne_commune_id_naissance=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("personne_nom")+";"+rs.getString("commune_nom")+";"+rs.getString("departement_numero");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
			}

			//Check bapteme
			requete=new StringBuffer();
			requete.append("select personne.personne_nom,commune.commune_nom,departement.departement_numero");
			requete.append(" from personne personne,commune commune, departement departement");
			requete.append(" where personne.personne_commune_id_bapteme=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("personne_nom")+";"+rs.getString("commune_nom")+";"+rs.getString("departement_numero");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
			}

			//Check deces
			requete=new StringBuffer();
			requete.append("select personne.personne_nom,commune.commune_nom,departement.departement_numero");
			requete.append(" from personne personne,commune commune, departement departement");
			requete.append(" where personne.personne_commune_id_deces=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("personne_nom")+";"+rs.getString("commune_nom")+";"+rs.getString("departement_numero");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
			}

			//Check inhumation
			requete=new StringBuffer();
			requete.append("select personne.personne_nom,commune.commune_nom,departement.departement_numero");
			requete.append(" from personne personne,commune commune, departement departement");
			requete.append(" where personne.personne_commune_id_inhumation=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("personne_nom")+";"+rs.getString("commune_nom")+";"+rs.getString("departement_numero");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
			}

			//Check mariage civil
			requete=new StringBuffer();
			requete.append("select personne1.personne_nom as mari,personne2.personne_nom as femme,");
			requete.append("commune.commune_nom as comm,departement.departement_numero as dno");
			requete.append(" from mariage mariage,personne personne1,personne personne2,commune commune,departement departement");
			requete.append(" where personne1.personne_id=mariage.mariage_mari_id");
			requete.append(" and personne2.personne_id=mariage.mariage_femme_id");
			requete.append(" and mariage.mariage_commune_id_civil=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("mari")+";"+rs.getString("comm")+";"+rs.getString("dno");
				String chaine1=rs.getString("femme")+";"+rs.getString("comm")+";"+rs.getString("dno");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
				if( noms.get(chaine1)==null )
					noms.put(chaine1,"");
			}

			//Check mariage religieux
			requete=new StringBuffer();
			requete.append("select personne1.personne_nom as mari,personne2.personne_nom as femme,");
			requete.append("commune.commune_nom as comm,departement.departement_numero as dno");
			requete.append(" from mariage mariage,personne personne1,personne personne2,commune commune, departement departement");
			requete.append(" where personne1.personne_id=mariage.mariage_mari_id");
			requete.append(" and personne2.personne_id=mariage.mariage_femme_id");
			requete.append(" and mariage.mariage_paroisse_id_religieux=commune.commune_id");
			requete.append(" and commune.commune_departement_id=departement.departement_id");
			ps=conn.prepareStatement(requete.toString());
			rs=ps.executeQuery();
			while( rs.next() ) {
				String chaine=rs.getString("mari")+";"+rs.getString("comm")+";"+rs.getString("dno");
				String chaine1=rs.getString("femme")+";"+rs.getString("comm")+";"+rs.getString("dno");
				if( noms.get(chaine)==null )
					noms.put(chaine,"");
				if( noms.get(chaine1)==null )
					noms.put(chaine1,"");
			}
		} catch(SQLException ex) {
			Log.log(ex);
			ex.printStackTrace();
		} finally {
			try {
				if( rs!=null )
					rs.close();
				if( ps!=null )
					ps.close();
			} catch(SQLException ex) {
			}
		}
		return noms;
	}
}