/** Requiert util.js */

/** @class
 * Creer une suggestion de saisie pour un champ textuel
 * @param idField : id du champ de saisie
 * @param getValuesFunction : fonction des parametres 
 * (unTexte, nbResultats) qui renvoie un tableau JavaScript 
 * des valeurs correspondant a la saisie unTexte, ce tableau
 * etant limite aux nbResultats premieres valeurs
 * @param maxSuggestNumber (optionnel) : nombre maximal de 
 * resultats a afficher (defaut : 10)
 */
function Suggest(idField, getValuesFunction, maxSuggestNumber) {
  /** Le champ de saisie @type HTMLInputElement */
  this.source = document.getElementById(idField);
  /** La fonction rcuperant les valeurs @type Function*/
  this.getValues = getValuesFunction;
  /** Nombre maximum de valeurs suggeres @type int */
  this.maxSuggestNumber = (maxSuggestNumber) ? maxSuggestNumber : 10;
  // Verifier la validit des paramtres
  this.check(idField);
  /** La zone de suggestion @type PopupList */
  this.popup = new PopupList(this.source);
  /** Valeur saisie par l'utilisateur @type String */
  this.inputValue = "";
  this.setBehaviour();
}

Suggest.prototype = {
  /** Vrifier que les paramtres sont valides */
  check: function(idField) {
    // Verifier qu'il y a bien une saisie a suggerer
    if (this.source == null) {
      Log.error("Element with id '" + idField + "' not found");
    }
    if (typeof(this.getValues) != "function") {
      Log.error("Suggestion function for '" + 
        idField + "' not found");
    }
    if (isNaN(parseInt(this.maxSuggestNumber)) ||
        parseInt(this.maxSuggestNumber) <= 0) {
      Log.error("Max suggest number for '" + idField + 
        "' not positive (" + this.maxSuggestNumber + ")");
    }
  },

  /** Dfinir les ractions du champ de saisie */
  setBehaviour: function() {
    // Desactiver la completion automatique du navigateur
    this.source.setAttribute("autocomplete", "off");
    // Stocker l'objet courant ...
    var suggest = this;
    // ... car dans la fonction ci-dessous, this est 
    // le champ de saisie (this.source) qui a genere l'evenement
    this.source.onkeyup = function(aEvent) {
      suggest.onkeyup(aEvent);
    };
    // Gerer l'evenement keydown qui est lance AVANT keyup,
    // or si on fait ENTER, par defaut le formulaire est soumis :
    // on ne peut pas bloquer cela dans onkeyup
    this.source.onkeydown = function(aEvent) {
      suggest.onkeydown(aEvent);
    }
    this.source.onblur = function() {
      // Masquer la popup seulement si la souris n'y est pas
      // Comme le mouseout est declenche avant le clic ou le tab
      // qui fait perdre le focus, il n'y a plus de div selectionne
      if (suggest.popup.index == -1) {
        suggest.popup.hide();
      }
    };
  },

  /** Raction  keydown */
  onkeydown: function(aEvent) {
    var event = Event.event(aEvent);
    switch (event.keyCode) {
      case Keys.ESCAPE:
        this.popup.hide();
        break;
      // S'il y a une suggestion, l'efface
      // s'il n'y en a pas (ou plus), soumet le formulaire
      case Keys.ENTER:
        if (this.popup.isVisible()) {
          Event.preventDefault(event);
          this.popup.clear();
        }
        break;
      case Keys.TAB:
        this.popup.clear();
        break;
      case Keys.DOWN:
        this.goAndGet(this.popup.index + 1);
        break;
      case Keys.UP:
        this.goAndGet(this.popup.index - 1);
        break;
      case Keys.PAGE_UP:
        this.goAndGet((this.popup.getLength() > 0) ? 0 : -1);
        break;
      case Keys.PAGE_DOWN:
        this.goAndGet(this.popup.getLength() - 1);
        break;
      default:
        break;
    }
  },

  /** Raction  la saisie (onkeyup) dans le champ */
  onkeyup: function(aEvent) {
    // L'evenement selon W3 ou IE
    switch (Event.event(aEvent).keyCode) {
      // Ne rien faire pour les touches de navigation
      // qui sont prises en compte par keydown
      case Keys.DOWN: case Keys.UP: case Keys.PAGE_UP: 
      case Keys.HOME: case Keys.PAGE_DOWN: case Keys.END:
      case Keys.ENTER: case Keys.ESCAPE:
        break;
      default:
        if (this.source.value != this.inputValue) {
          // Memoriser la saisie
          this.inputValue = this.source.value;
          // Mettre a jour la liste
          this.setOptions(this.source.value, this.maxSuggestNumber);
        }
    }
  },

  /** Rcuperer les options et les faire afficher */
  setOptions: function() {
    var values = this.getValues(this.source.value, 
      this.maxSuggestNumber);
    this.popup.setOptions(values);
  },
  
  /** Aller  la suggestion d'indice index
   * et mettre sa valeur dans le champ de saisie */
  goAndGet: function(index) {
    this.popup.go(index);
    if (-1 < this.popup.index) {
      this.source.value = this.popup.getValue();
    }
    else {
      this.source.value = this.inputValue;
    }
  }
}

Suggest.prototype.constructor = Suggest;