/** @class
 * Mise  jour d'une liste droulante par des appels Ajax
 * @requires json.js (String.prototype.parseJSON, 
 * Object.prototype.toJSONString et Array.prototype.toJSONString)
 * @param idSelect id de la liste droulante @type String
 * @param getOptionsUrl url de l'action serveur donnant les
 * options correspondant  une valeur passeen paramtre
 * @type String
 * @param idMsg id de l'lment HTML devant recevoir les
 * messages (transfert en cours, erreurs) @type String
 */
function SelectUpdater(idSelect, getOptionsUrl, idMsg) {
  /** Liste droulante  mettre  jour @type HTMLSelectElement */
  this.select = document.getElementById(idSelect);
  this.select.disabled = true;
  // Vrifier que le id est ok
  if (!this.select) {
    Log.error("Erreur sur new SelectUpdater('" + idSelect
      + " ' ...) : " + idSelect + " introuvable");
  }
  /** Url de la requte XMLHttpRequest mettant  jour @type String */
  this.url = getOptionsUrl;
  /** Requte XMLHttpRequest de mise  jour @type XMLHttpRequest */
  this.request = null;
  /** Message si aucune option n'est trouve 
   * (dfaut : "Valeur inconnue") @type String
   */
  this.NOT_FOUND_MSG = "Valeur inconnue";
  /** Elment HTML montrant le message @type HTMLElement*/
  this.msg = document.getElementById(idMsg);
  if (!this.msg) {
    Log.error("Erreur sur new SelectUpdater(..., ..., '"
      + idMsg + "') : " + idMsg + " introuvable");
  }
}

SelectUpdater.prototype = {
  /** Lancer la requte
   * @param value valeur passe  la requte @type String
   */
  run: function(value) {
    if (this.request) {
      try {
        this.request.abort();
      }
      catch (exc) {}
    }
    try {
      this.request = new XMLHttpRequest();
      var url = this.url + encodeURIComponent(value);
      this.request.open("GET", url, true);
      this.show();
      var current = this;
      this.request.onreadystatechange = function() {
        try {
          if (current.request.readyState == 4) {
            if (current.request.status == 200) {
              current.onload();
            }
            else {
              current.msg.innerHTML = "Erreur HTTP " 
                + current.request.status + " sur '" 
                + current.url + "'";
            }
          }
        }
        catch (exc) {
        }
      }
      this.request.send("");
    }
    catch (exc) {
      Log.debug(exc);
    }
  },
  
  /** Mettre  jour la liste  la rception de la rponse.
   * Celle-ci peut tre de type texte (ou html), XML ou JSON.
  */
  onload: function() {
    var hasContent, loadContent;
    var type = 
      this.request.getResponseHeader("Content-Type").split(";")[0];
    switch (type) {
      case "text/plain":case "text/html":
        hasContent = (this.request.responseText != "");
        loadContent = this.loadText;
        break;
      case "text/javascript":case "application/json":
        var result = this.request.responseText.parseJSON();
        try {
          hasContent = (result.items.length != 0);
        }
        catch (exc) {
          hasContent = false;
          Log.error(this.url + " ne renvoie pas un tableau JSON");
        }
        loadContent = this.loadJSON;
        break;
      case "text/xml":case "application/xml":
        var root = this.request.responseXML.documentElement;
        hasContent = (root.childNodes.length > 0);
        loadContent = this.loadXML;
        break;
      defaut:
        Log.error(this.url + " has an invalid Content-Type." +
          "Found '" + type + "' (must be plain text, JSON or XML)");
    }
    // Traitement commun
    this.select.innerHTML = "";
    this.hide();
    if (hasContent) {
      loadContent.apply(this);
      this.select.disabled = false;
    }
    else {
      this.msg.innerHTML = "<span style='color: red'>"
        + this.NOT_FOUND_MSG + "</span>";
      this.select.disabled = true;
    }
  },
  
  /** Crer les options quand la rponse est de type texte.
   * Elle est suppose alors de la forme <code>
   * value=text;value=text;...value=text</code>
   */
  loadText: function() {
    var options = this.request.responseText.split(";");
    var item, option;
    for (var i=0 ; i<options.length ; i++) {
      item = options[i].split("="); // value = text
      option = document.createElement("option");
      option.setAttribute("value", item[0]);
      option.innerHTML = item[1];
      this.select.appendChild(option);
    }
  },

  /** Crer les options quand la rponse est de type JSON.
   * Elle est suppose de la forme <code>
   * [{"value":"...", "text": "..."}, etc.]</code>
   */
  loadJSON: function() {
    var items = this.request.responseText.parseJSON().items;
    var item, option;
    for (var i=0 ; i<items.length ; i++) {
      option = document.createElement("option");
      option.setAttribute("value", items[i].value);
      option.innerHTML = items[i].text;
      this.select.appendChild(option);
    }
  },
  
  /** Crer les options quand la rponse est de type XML 
   * La rponse est suppose de la forme <pre>
   * &lt;unNomQuelconque attributsQuelconques="..."&gt;
   *   &lt;item value="codeItem" text="contenuItem"/&gt;
   *   &lt;item value="codeItem" text="contenuItem"/&gt;
   *   etc.<br/>&lt;/unNomQuelconque&gt; </pre>
   * Si ce n'est pas le cas, appelle la mthode transformXML
   * qui par defaut renvoie la rponse XML, sans transformation.
   */
  loadXML: function() {
    var root = this.request.responseXML.documentElement;
    var items = root.childNodes;
    var option;
    for (var i=0 ; i < items.length ; i++) {
      if (items[i].nodeName == "item") {
        option = document.createElement("option");
        option.setAttribute("value", 
          items[i].getAttribute("value"));
        option.innerHTML = items[i].getAttribute("text");
        this.select.appendChild(option);
      }
    }
  },
  
  /** Montrer que l'appel est en cours */
  show: function() {
    this.msg.innerHTML = "<em>En chargement ...</em>";
  },
  
  /** Effacer le message */
  hide: function() {
    this.msg.innerHTML = "";
  },
  
  /** Effacer la liste et le message, et annuler l'appel ventuel */
  reset: function() {
    this.select.innerHTML = "";
    this.select.disabled = true;
    this.msg.innerHTML = "";
    try {
      if (this.request) {
        this.request.abort();
      }
    }
    catch (exc) {}
  }
}
