/*
  OBJECTIF:
    Contrler la souris via DirectInput. Son curseur est une image charge dans
    une surface offscreen qui est recopie dans le backbuffer via
    IDirectDrawSurface4::BltFast () pour aller plus vite.

    Il faudrait en thorie recharger l'image dans la surface offscreen quand la
    mthode Restore () est appelle car rien ne garantie que, si cette surface
    est perdue puis restaure, son contenu soit prserv.

  INTERFACE:
    'g': Lancer la dmonstration
    'ESC': Quitter la dmonstration

  NOTES:
    1/ Crer l'objet DirectInput via DirectInputCreate ()

    2/ Crer l'objet Device via IDirectInput::CreateDevice (). Cet objet
       contrle un "Direct Input device" identifi par un GUID, par exemple
       GUID_SysMouse pour la souris

    3/ Indiquer via IDirectInputDevice::SetDataFormat () quelles donnes l'objet
       Device doit retourner et comment il doit les retourner. Il s'agit en fait
       de dcrire une structure T, ventuellement prdfinie (c_dfDIMouse).
 
       Chaque champ de la structure T est associe  une donne que retourne
       l'objet Device et est dcrit par une structure DIOBJECTDATAFORMAT:

       typedef struct _DIOBJECTDATAFORMAT
       {
         const GUID *pguid;
         DWORD dwOfs;
         DWORD dwType;
         DWORD dwFlags;
       } DIOBJECTDATAFORMAT, *LPDIOBJECTDATAFORMAT;

       const GUID *pguid
         Pointeur sur le GUID identifiant la source de la donne que l'objet
         Device doit retourner (GUID_XAxis, GUID_YAxis, ...)

       DWORD dwOfs
         Offset du champ dans lequel doit tre stocke cette donne dans la
         structure

       DWORD dwType
         Une constante DIDFT_x dcrivant la source de donne

       DWORD dwFlags
         Une constante DIDOI_x, mais plus gnralement 0

       Attention ! Certains "Direct Input device" stockent les donnes qui leurs
       sont demandes dans des champs qui doivent tre d'une certaine taille et
       respecter un certain alignement. La taille de la structure T doit tre
       multiple de 4.

       Les structures DIOBJECTDATAFORMAT doivent tre rassembles dans un
       tableau. Une structure DIDATAFORMAT sert alors  dcrire la structure T:

       typedef struct _DIDATAFORMAT
       {
         DWORD   dwSize;
         DWORD   dwObjSize;
         DWORD   dwFlags;
         DWORD   dwDataSize;
         DWORD   dwNumObjs;
         LPDIOBJECTDATAFORMAT rgodf;
       } DIDATAFORMAT, *LPDIDATAFORMAT;

       DWORD dwObjSize
         sizeof (DIOBJECTDATAFORMAT)

       DWORD dwFlags
         Une constante DIDF_*

       DWORD dwDataSize
         La taille de la structure T

       DWORD dwNumObjs
         Nombre de structure DIOBJECTDATAFORMAT  suivre

       LPDIOBJECTDATAFORMAT rgodf
         Pointeur sur le tableau des structures DIOBJECTDATAFORMAT

    4/ Indiquer le niveau de coopration via IDirectInputDevice::SetCooperativeLevel ()

    5/ Obtenir l'tat de la souris. Il y a deux moyens:

       a/ Cooprer avec Windows:
       
          1/ Utiliser IDirectInputDevice::SetProperty () pour fournir un buffer
              l'objet Device. Ce dernier y stockera les structures T sans en
             manquer une.
           
          2/ Crer un venement via la fonction CreateEvent () de Win32
          
          3/ Utiliser IDirectInputDevice::SetEventNotification () pour associer
             l'objet Device  cet venement

          4/ Modifier la boucle de messages de l'application:

             {
               Attendre un message via MsgWaitForMultipleObjects() ou un message standard
               Acqurir le contrle de la souris via IDirectInputDevice::Acquire ()
               Obtenir toutes les structures T accumules via IDirectInputDevice::GetDeviceData ()
               Librer la souris via IDirectInputDevice::Unacquire ()
               Obtenir et dispatcher les messages standard
             }
          
          Cette mthode permet de ne perdre aucune action des actions de
          l'utilisateur mais n'en permet pas de la gestion en temps rel.

       b/ Faire une boucle propritaire:
       
          1/ Acqurir le contrle de la souris via IDirectInputDevice::Acquire ()

          2/ Faire une boucle:
 
             {
               Obtenir la structure T courante via IDirectInputDevice::GetDeviceState ()
             }
    
          3/ Librer la souris via IDirectInputDevice::Unacquire ()
          
          Cette mthode permet de ragir en temps rel aux actions de
          l'utilisateur mais fait perdre le contrle de la souris  Windows.

  Copyright (c) 1999 Denis Duplan. Tous droits rservs.
*/

#define INITGUID
#include <windows.h>
#include <stdio.h>
#include <memory.h>
#include <ddraw.h>
#include <dinput.h>

/* CONSTANTES ----------------------------------------------------------------*/

#define SCREEN_DX 800
#define SCREEN_DY 600

#define BITMAP_FILE "data\\test.bmp"

/* CLASSES -------------------------------------------------------------------*/

class TDDDemo
{
  public:
    HWND _hWindow;
    LPDIRECTDRAW4 _lpDD4Interface;
    LPDIRECTDRAWSURFACE4 _lpDDS4Primary,
                         _lpDDS4Backbuffer,
                         _lpDDS4Mouse;
    LPDIRECTINPUT _lpDIInterface;
    LPDIRECTINPUTDEVICE _lpDIDMouse;

    TDDDemo (HWND, HINSTANCE, BOOL *);
    BOOLEAN blMouseSetup (void);
    void MouseLoop (void);
    BOOL blMouseDraw (WORD, WORD);
    void Flip (void);
    void Restore (void);
    ~TDDDemo (void);
};

/* TYPES ---------------------------------------------------------------------*/

/* Variables globales */

typedef struct
{
  HINSTANCE _hInstance;
  TDDDemo *_pDDDemo;
  BOOL _blActive,
       _blValid;
} TGlobals;

/* Structure permettant de rcuprer l'tat de la souris */

typedef struct
{
  LONG _lX;          /* Axe. 4 octets, aligne sur 4 octets */
  LONG _lY;          /* Axe. 4 octets, aligne sur 4 octets */
  BYTE _bLButton,    /* Bouton. 1 octet, align sur 1 octet */
       _bRButton,    /* Bouton. 1 octet, align sur 1 octet */
       _bPadding[2]; /* La taille de la structure doit tre multiple de 4 */
} TMouseState;

/* GLOBAL VARIABLES ----------------------------------------------------------*/

TGlobals Globals;

/******************************************************************************/
/*                                                                            */
/*                                   TDDemo                                   */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Crer un objet DirectDraw                                                */
/*   Changer le mode vido en SCREEN_DX x SCREEN_DY, 16 bits                  */
/*   Crer la surface primaire et son backbuffer                              */
/*   Crer une surface offscreen et y charger l'image du curseur              */
/*   Prendre le contrle de la souris                                         */
/*                                                                            */
/* ENTREE:                                                                    */
/*   HWND hWindow                                                             */
/*     Handle de la fentre principale de l'application                       */
/*   HINSTANCE hInstance                                                      */
/*     Handle de l'instance de l'application                                  */
/*   BOOL *pblValid                                                           */
/*     Pointeur sur un boolen qui contiendra TRUE si l'initialisation s'est  */
/*     bien droule et FALSE dans le cas contraire                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::TDDDemo (HWND hWindow, HINSTANCE hInstance, BOOL *pblValid)
{ 
  LPDIRECTDRAW lpDDInterface;
  DDSCAPS2 DDSC2Caps;
  DDSURFACEDESC2 DDSD2Desc;
  HRESULT HResult;
  HDC BitmapDC,
      SurfaceDC;
  HBITMAP BitmapHandle;
  BITMAP Bitmap;

  /* Valeur retourne par dfaut */

  *pblValid = FALSE;

  /* Initialiser l'objet */

  _hWindow = hWindow;
  _lpDD4Interface = NULL;
  _lpDDS4Backbuffer = NULL;
  _lpDDS4Primary = NULL;
  _lpDDS4Mouse = NULL;

  /* Crer un objet DirectDraw en utilisant le "DirectDraw device driver" par dfaut */

  HResult = DirectDrawCreate (NULL,
                              &lpDDInterface,
                              NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDraw4 */

  HResult = lpDDInterface->QueryInterface (IID_IDirectDraw4,
                                           (LPVOID *) &_lpDD4Interface);
  if (HResult != S_OK)
    return;
    
  /* Librer l'interface IDirectDraw */

  if (lpDDInterface->Release () != 0)
    return;

  /* Indiquer le niveau de coopration */

  HResult = _lpDD4Interface->SetCooperativeLevel (_hWindow,
                                                  DDSCL_EXCLUSIVE |
                                                  DDSCL_FULLSCREEN);
  if (HResult != DD_OK)
    return;

  /* Changer de mode vido */

  HResult = _lpDD4Interface->SetDisplayMode (SCREEN_DX,
                                             SCREEN_DY,
                                             16,
                                             0,
                                             0);
  if (HResult != DD_OK)
    return;

  /* Crr la surface primaire et son backbuffer */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  DDSD2Desc.dwFlags = DDSD_CAPS |
                      DDSD_BACKBUFFERCOUNT;
  DDSD2Desc.dwBackBufferCount = 1;
  DDSD2Desc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                             DDSCAPS_COMPLEX |
                             DDSCAPS_FLIP;
  HResult = _lpDD4Interface->CreateSurface (&DDSD2Desc,
                                            &_lpDDS4Primary,
                                            NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDrawSurface4 du backbuffer */

  DDSC2Caps.dwCaps = DDSCAPS_BACKBUFFER;
  HResult = _lpDDS4Primary->GetAttachedSurface (&DDSC2Caps,
                                                &_lpDDS4Backbuffer);
  if (HResult != DD_OK)
    return;

  /* Dcompresser les donnes du fichier BMP */

  BitmapHandle = (HBITMAP) LoadImage (NULL,
                                      BITMAP_FILE,
                                      IMAGE_BITMAP,
                                      0, 0,
                                      LR_LOADFROMFILE);
  if (BitmapHandle == NULL)
    return;
  if (GetObject (BitmapHandle, sizeof (BITMAP), &Bitmap) == 0)
    return;

  /* Crer une surface offscreen */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  DDSD2Desc.dwFlags = DDSD_CAPS |
                      DDSD_WIDTH |
                      DDSD_HEIGHT |
                      DDSD_PIXELFORMAT;
  DDSD2Desc.dwWidth = Bitmap.bmWidth;
  DDSD2Desc.dwHeight = Bitmap.bmHeight;
  DDSD2Desc.ddpfPixelFormat.dwSize = sizeof (DDPIXELFORMAT);
  DDSD2Desc.ddpfPixelFormat.dwFlags = DDPF_RGB;
  DDSD2Desc.ddpfPixelFormat.dwRGBBitCount = 16;
  DDSD2Desc.ddpfPixelFormat.dwRBitMask = 0xF800;
  DDSD2Desc.ddpfPixelFormat.dwGBitMask = 0x07E0;
  DDSD2Desc.ddpfPixelFormat.dwBBitMask = 0x001F;
  DDSD2Desc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
  HResult = _lpDD4Interface->CreateSurface (&DDSD2Desc,
                                            &_lpDDS4Mouse,
                                            NULL);
  if (HResult != DD_OK)
    return;
  
  /* Recopier le contenu du fichier BMP dans la surface offscreen */

  BitmapDC = CreateCompatibleDC (NULL);
  if (BitmapDC == NULL)
    return;
  SelectObject (BitmapDC, BitmapHandle);
  HResult = _lpDDS4Mouse->GetDC (&SurfaceDC);
  if (HResult != DD_OK)
    return;
  BitBlt (SurfaceDC,
          0, 0,
          Bitmap.bmWidth,
          Bitmap.bmHeight,
          BitmapDC,
          0, 0,
          SRCCOPY);
  HResult = _lpDDS4Mouse->ReleaseDC (SurfaceDC);
  if (HResult != DD_OK)
    return;
  DeleteDC (BitmapDC);

  /* Dcharger l'image */

  DeleteObject (BitmapHandle);

  /* Prendre le contrle de la souris */

  if (blMouseSetup () == FALSE)
    return;

  /* Boucle */

  MouseLoop ();

  /* Valider l'initialisation */

  *pblValid = TRUE;

  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Prendre le contrle de la souris                                         */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   FALSE si une erreur survient, sinon TRUE                                 */
/*----------------------------------------------------------------------------*/

BOOLEAN TDDDemo::blMouseSetup (void)
{
  HRESULT HResult;
  DIOBJECTDATAFORMAT DIODFMouse[] = {
                                      {
                                        &GUID_XAxis,
                                        FIELD_OFFSET (TMouseState, _lX),
                                        DIDFT_AXIS | DIDFT_ANYINSTANCE,
                                        0
                                      },
                                      {
                                        &GUID_YAxis,
                                        FIELD_OFFSET (TMouseState, _lY),
                                        DIDFT_AXIS | DIDFT_ANYINSTANCE,
                                        0
                                      },
                                      {
                                        &GUID_Button,
                                        FIELD_OFFSET (TMouseState, _bLButton),
                                        DIDFT_BUTTON | DIDFT_ANYINSTANCE,
                                        0
                                      },
                                      {
                                        &GUID_Button,
                                        FIELD_OFFSET (TMouseState, _bRButton),
                                        DIDFT_BUTTON | DIDFT_ANYINSTANCE,
                                        0
                                      }
                                    };
  DIDATAFORMAT DIDFMouse = {
                             sizeof (DIDATAFORMAT),
                             sizeof (DIOBJECTDATAFORMAT),
                             DIDF_RELAXIS,
                             sizeof (TMouseState),
                             4,
                             DIODFMouse
                           };

  /* Crer un objet DirectInput */

  HResult = DirectInputCreate (Globals._hInstance,
                               DIRECTINPUT_VERSION,
                               &_lpDIInterface,
                               NULL);
  if (HResult != DI_OK)
    return (FALSE);

  /* Crer un objet Device */

  HResult = _lpDIInterface->CreateDevice (GUID_SysMouse,
                                          &_lpDIDMouse,
                                          NULL);
  if (HResult != DI_OK)
    return (FALSE);

  /* Donner la description de la structure TMouseState */

  _lpDIDMouse->SetDataFormat (&DIDFMouse);
  if (HResult != DI_OK)
    return (FALSE);

  /* Indiquer le niveau de coopration */
  
  HResult = _lpDIDMouse->SetCooperativeLevel (_hWindow,
                                              DISCL_FOREGROUND |
                                              DISCL_EXCLUSIVE);
  if (HResult != DI_OK)
    return (FALSE);

  return (TRUE);
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Boucle principale                                                        */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

void TDDDemo::MouseLoop (void)
{
  HRESULT HResult;
  TMouseState MouseState;
  LONG lMouseX, lMouseY,
       lOldMouseX, lOldMouseY;

  /* Afficher le curseur de la souris en (0, 0) */

  if (blMouseDraw (0, 0) == FALSE)
    return;

  /* Acqurir le contrle de la souris */

  HResult = _lpDIDMouse->Acquire ();
  if (HResult != DI_OK)
    return;

  /* Initialisations */

  lMouseX = 0;
  lMouseY = 0;
  lOldMouseX = 0;
  lOldMouseY = 0;

  /* Boucle DirectInput. Attention: elle peut tre interrompue par Windows */

  while (1)
  {
    /* Obtenir l'tat de la souris */

    HResult = _lpDIDMouse->GetDeviceState (sizeof (TMouseState),
                                           &MouseState);
    if (HResult != DI_OK)
      return;

    /* Tester le bouton gauche de la souris */

    if (MouseState._bLButton != 0)
      break;

    /* Changer la position de la souris */

    lMouseX += MouseState._lX;
    if (lMouseX > (SCREEN_DX - 1))
      lMouseX = SCREEN_DX - 1;
    if (lMouseX < 0)
      lMouseX = 0;
    lMouseY += MouseState._lY;
    if (lMouseY > (SCREEN_DY - 1))
      lMouseY = SCREEN_DY - 1;
    if (lMouseY < 0)
      lMouseY = 0;

    /* Vrifier si la souris s'est dplac */

    if ((lMouseX != lOldMouseX) || (lMouseY != lOldMouseY))
    {
      /* Attendre un VBL */
  
      _lpDD4Interface->WaitForVerticalBlank (DDWAITVB_BLOCKBEGIN,
                                             NULL);

      /* Afficher le curseur de la souris  sa nouvelle position */

      if (blMouseDraw (lMouseX, lMouseY) == FALSE)
        return;
      lOldMouseX = lMouseX;
      lOldMouseY = lMouseY;
    }
  }  

  /* Librer la souris */

  _lpDIDMouse->Unacquire ();

  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Afficher le curseur de la souris                                         */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

BOOL TDDDemo::blMouseDraw (WORD wX, WORD wY)
{
  DDSURFACEDESC2 DDSD2Desc;
  RECT Rect;
  HRESULT HResult;
  DDBLTFX DDBFBlitFX;

  /* Obtenir les dimensions de la surface offscreen */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  HResult = _lpDDS4Mouse->GetSurfaceDesc (&DDSD2Desc);
  if (HResult != DD_OK)
    return (FALSE);
  Rect.top = 0;
  Rect.left = 0;
  Rect.right = DDSD2Desc.dwWidth - 1;
  Rect.bottom = DDSD2Desc.dwHeight - 1;

  /* Calculer la rgion du curseur  afficher */

  Rect.left = 0;
  Rect.top = 0;

  if ((wX + DDSD2Desc.dwWidth) > SCREEN_DX)
    Rect.right = SCREEN_DX - wX;
  else
    Rect.right = DDSD2Desc.dwWidth;
  if ((wY + DDSD2Desc.dwHeight) > SCREEN_DY)
    Rect.bottom = SCREEN_DY - wY;
  else
    Rect.bottom = DDSD2Desc.dwHeight;

  /* Effacer le contenu du backbuffer */

  memset (&DDBFBlitFX, 0, sizeof (DDBLTFX));
  DDBFBlitFX.dwSize = sizeof (DDBLTFX);
  HResult = _lpDDS4Backbuffer->Blt (NULL,
                                    NULL,
                                    NULL,
                                    DDBLT_WAIT | DDBLT_COLORFILL,
                                    &DDBFBlitFX);
  if (HResult != DD_OK)
    return (FALSE);

  /* Copier le curseur dans le backbuffer */

  HResult = _lpDDS4Backbuffer->BltFast (wX,
                                        wY,
                                        _lpDDS4Mouse,
                                        &Rect,
                                        DDBLTFAST_NOCOLORKEY);
  if (HResult != DD_OK)
    return (FALSE);

  /* Inverser la surface primaire et son backbuffer */

  HResult = _lpDDS4Primary->Flip (NULL,
                                  0);
  if (HResult != DD_OK)
    return (FALSE);

  return (TRUE);
}  

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Destructeur                                                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::~TDDDemo (void)
{
  HRESULT HResult;

  /* Restaurer le mode vido */

  HResult = _lpDD4Interface->RestoreDisplayMode ();
  if (HResult != DD_OK)
    return;

  /* Librer les interfaces */

  if (_lpDIDMouse != NULL)
    _lpDIDMouse->Release ();
  if (_lpDIInterface != NULL)
    _lpDIInterface->Release ();
  if (_lpDDS4Mouse != NULL)
    _lpDDS4Mouse->Release ();
  if (_lpDDS4Backbuffer != NULL)
    _lpDDS4Backbuffer->Release ();
  if (_lpDDS4Primary != NULL)
    _lpDDS4Primary->Release ();
  if (_lpDD4Interface != NULL)
    _lpDD4Interface->Release ();

  return;
}  

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Restaurer le niveau de coopration exclusif et les surfaces perdues      */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   FALSE si erreur, TRUE dans le cas contraire                              */
/*----------------------------------------------------------------------------*/

void TDDDemo::Restore (void)
{
  HRESULT HResult;

  /* Attendre le retour au mode coopratif exclusif */

  while (_lpDD4Interface->TestCooperativeLevel () != DD_OK)
    WaitMessage ();

  /* Restaurer les surfaces perdues */

  HResult = _lpDD4Interface->RestoreAllSurfaces ();
  if (HResult != DD_OK)
    return;

  return;
}

/******************************************************************************/
/*                                                                            */
/*                                 Application                                */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Fonction de rappel de la fentre principale                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

LRESULT CALLBACK WindowProc (HWND hWindow,
                             UINT uMessage,
                             WPARAM wParam,
                             LPARAM lParam)
{
  switch (uMessage)
  { 
    /* Pression d'une touche */

    case WM_CHAR:

      /* Si la touche est "g", lancer la dmonstration si elle ne l'est pas dj */

      if ((wParam == 'g') && (Globals._blValid == FALSE))
      {
        Globals._pDDDemo = new TDDDemo (hWindow,
                                        Globals._hInstance,
                                        &Globals._blValid);
        if (Globals._blValid == FALSE)
          delete (Globals._pDDDemo);
        break;
      }

      /* Si la dmonstration n'est pas lance, en rester l */

      if (Globals._blValid == FALSE)
        break;

      /* Tester les touches qui permettent de contrler le droulement de la dmonstration */  

      switch (wParam)
      {
        /* 'ESC': Quitter la dmonstration et l'application */

        case VK_ESCAPE:
          Globals._blValid = FALSE;
          delete (Globals._pDDDemo);
          DestroyWindow (hWindow);
          break;
      }
      break;

    /* Destruction de l'application */

    case WM_DESTROY:
      PostQuitMessage (0);
      break;

    /* Activation/dsactivation de l'application */

    case WM_ACTIVATEAPP:
      Globals._blActive = wParam;
      if ((Globals._blActive == FALSE) || (Globals._blValid == FALSE))
        break;
      Globals._pDDDemo->Restore ();
      break;
  }

  return (DefWindowProc (hWindow, uMessage, wParam, lParam));
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Point d'entre de l'application                                          */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int nCmdShow)
{
  HWND hWindow;
  WNDCLASS WClass;
  ATOM Atom;
  MSG Message;

  /* Initialiser les variables globales */

  Globals._hInstance = hInstance;
  Globals._pDDDemo = NULL;
  Globals._blValid = FALSE;

  /* Crer une classe de fentre */

  WClass.style = 0;
  WClass.lpfnWndProc = WindowProc;
  WClass.cbClsExtra = 0;
  WClass.cbWndExtra = 0;
  WClass.hInstance = hInstance;
  WClass.hIcon = NULL;
  WClass.hCursor = NULL;
  WClass.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
  WClass.lpszMenuName = NULL;
  WClass.lpszClassName = "DirectDraw";

  Atom = RegisterClass (&WClass);
  if (Atom == 0)
    return (0);

  /* Crer la fentre */

  hWindow = CreateWindow ("DirectDraw",
                          "Exemple DirectDraw",
                          WS_CAPTION |
                          WS_SYSMENU,
                          10,
                          50,
                          300,
                          200,
                          NULL,
                          NULL,
                          Globals._hInstance,
                          NULL);
  if (hWindow == 0)
    return (0);

  /* Montrer la fentre */

  ShowWindow (hWindow, nCmdShow);
  UpdateWindow (hWindow); 

  /* Boucle de messages */ 

  while (GetMessage (&Message, NULL, 0, 0) == TRUE)
  { 
    TranslateMessage (&Message); 
    DispatchMessage (&Message); 
  } 
  
  return (0);
}

