/*
  OBJECTIF:
    Verrouiller une surface pour accder directement  ses donnes et y tracer
    une droite.

  INTERFACE:
    'g': Lancer la dmonstration
    'f': Intervertir la surface primaire et son backbuffer
    'ESC': Quitter la dmonstration

  Copyright (c) 1999 Denis Duplan. Tous droits rservs.
*/

#include <windows.h>
#include <ddraw.h>

/* CONSTANTES ----------------------------------------------------------------*/

#define SCREEN_DX 800
#define SCREEN_DY 600

/* CLASSES -------------------------------------------------------------------*/

class TDDDemo
{
  public:
    HWND _hWindow;
    LPDIRECTDRAW4 _lpDD4Interface;
    LPDIRECTDRAWSURFACE4 _lpDDS4Primary,
                         _lpDDS4Backbuffer;

    TDDDemo (HWND, HINSTANCE, BOOL *);
    void Flip (void);
    void Restore (void);
    ~TDDDemo (void);
};

/* TYPES ---------------------------------------------------------------------*/

/* Variables globales */

typedef struct
{
  HINSTANCE _hInstance;
  TDDDemo *_pDDDemo;
  BOOL _blActive,
       _blValid;
} TGlobals;

/* GLOBAL VARIABLES ----------------------------------------------------------*/

TGlobals Globals;

/******************************************************************************/
/*                                                                            */
/*                                   TDDemo                                   */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Crer un objet DirectDraw                                                */
/*   Changer le mode vido en SCREEN_DX x SCREEN_DY, 16 bits                  */
/*   Crer la surface primaire et son backbuffer                              */
/*   Verrouiller le backbuffer et y tracer une droite                         */
/*                                                                            */
/* ENTREE:                                                                    */
/*   HWND hWindow                                                             */
/*     Handle de la fentre principale de l'application                       */
/*   HINSTANCE hInstance                                                      */
/*     Handle de l'instance de l'application                                  */
/*   BOOL *pblValid                                                           */
/*     Pointeur sur un boolen qui contiendra TRUE si l'initialisation s'est  */
/*     bien droule et FALSE dans le cas contraire                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::TDDDemo (HWND hWindow, HINSTANCE hInstance, BOOL *pblValid)
{ 
  LPDIRECTDRAW lpDDInterface;
  DDSCAPS2 DDSC2Caps;
  DDSURFACEDESC2 DDSD2Desc;
  HRESULT HResult;
  WORD *pwSurface,
       wCount;

  /* Valeur retourne par dfaut */

  *pblValid = FALSE;

  /* Initialiser l'objet */

  _hWindow = hWindow;
  _lpDD4Interface = NULL;
  _lpDDS4Backbuffer = NULL;
  _lpDDS4Primary = NULL;

  /* Crer un objet DirectDraw en utilisant le "DirectDraw device driver" par dfaut */

  HResult = DirectDrawCreate (NULL,
                              &lpDDInterface,
                              NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDraw4 */

  HResult = lpDDInterface->QueryInterface (IID_IDirectDraw4,
                                           (LPVOID *) &_lpDD4Interface);
  if (HResult != S_OK)
    return;
    
  /* Librer l'interface IDirectDraw */

  if (lpDDInterface->Release () != 0)
    return;

  /* Indiquer le niveau de coopration */

  HResult = _lpDD4Interface->SetCooperativeLevel (_hWindow,
                                                  DDSCL_EXCLUSIVE |
                                                  DDSCL_FULLSCREEN);
  if (HResult != DD_OK)
    return;

  /* Changer de mode vido */

  HResult = _lpDD4Interface->SetDisplayMode (SCREEN_DX,
                                             SCREEN_DY,
                                             16,
                                             0,
                                             0);
  if (HResult != DD_OK)
    return;

  /* Crr la surface primaire et son backbuffer */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  DDSD2Desc.dwFlags = DDSD_CAPS |
                      DDSD_BACKBUFFERCOUNT;
  DDSD2Desc.dwBackBufferCount = 1;
  DDSD2Desc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                             DDSCAPS_COMPLEX |
                             DDSCAPS_FLIP;
  HResult = _lpDD4Interface->CreateSurface (&DDSD2Desc,
                                            &_lpDDS4Primary,
                                            NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDrawSurface4 du backbuffer */

  DDSC2Caps.dwCaps = DDSCAPS_BACKBUFFER;
  HResult = _lpDDS4Primary->GetAttachedSurface (&DDSC2Caps,
                                                &_lpDDS4Backbuffer);
  if (HResult != DD_OK)
    return;

  /* Verrouiller le backbuffer */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  HResult = _lpDDS4Backbuffer->Lock (NULL,
                                     &DDSD2Desc,
                                     DDLOCK_SURFACEMEMORYPTR,
                                     NULL);
  if (HResult != DD_OK)
    return;

  /* Dessiner une droite horizontale sur un fond noir */

  pwSurface = (WORD *) DDSD2Desc.lpSurface;
  for (wCount = 0; wCount != SCREEN_DY; wCount ++)
  {
    memset (pwSurface, 0, SCREEN_DX << 1);
    *(pwSurface + (SCREEN_DX >> 1)) = 0xFFFF;
    pwSurface += (DDSD2Desc.lPitch >> 1);
  }

  /* Dverrouiller le backbuffer */

  HResult = _lpDDS4Backbuffer->Unlock (NULL);
  if (HResult != DD_OK)
    return;

  /* Valider l'initialisation */

  *pblValid = TRUE;
                                   
  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Inverser la surface primaire et son backbuffer                           */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

void TDDDemo::Flip (void)
{
  HRESULT HResult;

  /* Inverser la surface primaire et son backbuffer */

  HResult = _lpDDS4Primary->Flip (NULL,
                                  0);
  if (HResult != DD_OK)
    return;

  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Destructeur                                                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::~TDDDemo (void)
{
  HRESULT HResult;

  /* Restaurer le mode vido */

  HResult = _lpDD4Interface->RestoreDisplayMode ();
  if (HResult != DD_OK)
    return;

  /* Librer les interfaces */

  if (_lpDDS4Backbuffer != NULL)
    _lpDDS4Backbuffer->Release ();
  if (_lpDDS4Primary != NULL)
    _lpDDS4Primary->Release ();
  if (_lpDD4Interface != NULL)
    _lpDD4Interface->Release ();

  return;
}  
  
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Restaurer le niveau de coopration exclusif et les surfaces perdues      */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   FALSE si erreur, TRUE dans le cas contraire                              */
/*----------------------------------------------------------------------------*/

void TDDDemo::Restore (void)
{
  HRESULT HResult;

  /* Attendre le retour au mode coopratif exclusif */

  while (_lpDD4Interface->TestCooperativeLevel () != DD_OK)
    WaitMessage ();

  /* Restaurer les surfaces perdues */

  HResult = _lpDD4Interface->RestoreAllSurfaces ();
  if (HResult != DD_OK)
    return;

  return;
}

/******************************************************************************/
/*                                                                            */
/*                                 Application                                */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Fonction de rappel de la fentre principale                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

LRESULT CALLBACK WindowProc (HWND hWindow,
                             UINT uMessage,
                             WPARAM wParam,
                             LPARAM lParam)
{
  switch (uMessage)
  { 
    /* Pression d'une touche */

    case WM_CHAR:

      /* Si la touche est "g", lancer la dmonstration si elle ne l'est pas dj */

      if ((wParam == 'g') && (Globals._blValid == FALSE))
      {
        Globals._pDDDemo = new TDDDemo (hWindow,
                                        Globals._hInstance,
                                        &Globals._blValid);
        if (Globals._blValid == FALSE)
          delete (Globals._pDDDemo);
        break;
      }

      /* Si la dmonstration n'est pas lance, en rester l */

      if (Globals._blValid == FALSE)
        break;

      /* Tester les touches qui permettent de contrler le droulement de la dmonstration */  

      switch (wParam)
      {
        /* 'ESC': Quitter la dmonstration et l'application */

        case VK_ESCAPE:
          Globals._blValid = FALSE;
          delete (Globals._pDDDemo);
          DestroyWindow (hWindow);
          break;

        /* 'f': Inverser la surface primaire et le backbuffer */
          
        case 'f':
          Globals._pDDDemo->Flip ();
          break;
      }
      break;

    /* Destruction de l'application */

    case WM_DESTROY:
      PostQuitMessage (0);
      break;

    /* Activation/dsactivation de l'application */

    case WM_ACTIVATEAPP:
      Globals._blActive = wParam;
      if ((Globals._blActive == FALSE) || (Globals._blValid == FALSE))
        break;
      Globals._pDDDemo->Restore ();
      break;
  }

  return (DefWindowProc (hWindow, uMessage, wParam, lParam));
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Point d'entre de l'application                                          */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int nCmdShow)
{
  HWND hWindow;
  WNDCLASS WClass;
  ATOM Atom;
  MSG Message;

  /* Initialiser les variables globales */

  Globals._hInstance = hInstance;
  Globals._pDDDemo = NULL;
  Globals._blValid = FALSE;

  /* Crer une classe de fentre */

  WClass.style = 0;
  WClass.lpfnWndProc = WindowProc;
  WClass.cbClsExtra = 0;
  WClass.cbWndExtra = 0;
  WClass.hInstance = hInstance;
  WClass.hIcon = NULL;
  WClass.hCursor = NULL;
  WClass.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
  WClass.lpszMenuName = NULL;
  WClass.lpszClassName = "DirectDraw";

  Atom = RegisterClass (&WClass);
  if (Atom == 0)
    return (0);

  /* Crer la fentre */

  hWindow = CreateWindow ("DirectDraw",
                          "Exemple DirectDraw",
                          WS_CAPTION |
                          WS_SYSMENU,
                          10,
                          50,
                          300,
                          200,
                          NULL,
                          NULL,
                          Globals._hInstance,
                          NULL);
  if (hWindow == 0)
    return (0);

  /* Montrer la fentre */

  ShowWindow (hWindow, nCmdShow);
  UpdateWindow (hWindow); 

  /* Boucle de messages */ 

  while (GetMessage (&Message, NULL, 0, 0) == TRUE)
  { 
    TranslateMessage (&Message); 
    DispatchMessage (&Message); 
  } 
  
  return (0);
}
