/*
  OBJECTIF:
    Afficher une image BMP dans une surface offscreen et blitter cette dernire
    dans le backbuffer. Restreindre cette fois ci le blit  certaines rgions
    du backbuffer grce  un objet Clipper. Notez qu'il n'est pas possible
    d'utiliser le service IDirectDrawSurface4::BltFast ().

  INTERFACE:
    'g': Lancer la dmonstration
    'f': Intervertir la surface primaire et son backbuffer
    'ESC': Quitter la dmonstration

  Copyright (c) 1999 Denis Duplan. Tous droits rservs.
*/

#include <windows.h>
#include <stdio.h>
#include <ddraw.h>

/* CONSTANTES ----------------------------------------------------------------*/

#define SCREEN_DX 800
#define SCREEN_DY 600

#define BITMAP_FILE "data\\test.bmp"

/* CLASSES -------------------------------------------------------------------*/

class TDDDemo
{
  public:
    HWND _hWindow;
    LPDIRECTDRAW4 _lpDD4Interface;
    LPDIRECTDRAWSURFACE4 _lpDDS4Primary,
                         _lpDDS4Backbuffer;

    TDDDemo (HWND, HINSTANCE, BOOL *);
    void Flip (void);
    void Restore (void);
    ~TDDDemo (void);
};

/* TYPES ---------------------------------------------------------------------*/

/* Variables globales */

typedef struct
{
  HINSTANCE _hInstance;
  TDDDemo *_pDDDemo;
  BOOL _blActive,
       _blValid;
} TGlobals;

/* Clip list pour l'objet Clipper */

typedef struct
{
  RGNDATAHEADER _Header;
  RECT _Rects[2];
} TClipList;

/* GLOBAL VARIABLES ----------------------------------------------------------*/

TGlobals Globals;

/******************************************************************************/
/*                                                                            */
/*                                   TDDemo                                   */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Crer un objet DirectDraw                                                */
/*   Changer le mode vido en SCREEN_DX x SCREEN_DY, 16 bits                  */
/*   Crer la surface primaire et son backbuffer                              */
/*   Crer une surface offscreen                                              */
/*   Charger l'image dans la surface offscreen                                */
/*   Crer un objet Clipper et l'associer au backbuffer                       */
/*   Recopier la surface offscreen dans le backbuffer                         */
/*                                                                            */
/* ENTREE:                                                                    */
/*   HWND hWindow                                                             */
/*     Handle de la fentre principale de l'application                       */
/*   HINSTANCE hInstance                                                      */
/*     Handle de l'instance de l'application                                  */
/*   BOOL *pblValid                                                           */
/*     Pointeur sur un boolen qui contiendra TRUE si l'initialisation s'est  */
/*     bien droule et FALSE dans le cas contraire                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::TDDDemo (HWND hWindow, HINSTANCE hInstance, BOOL *pblValid)
{ 
  LPDIRECTDRAW lpDDInterface;
  DDSCAPS2 DDSC2Caps;
  DDSURFACEDESC2 DDSD2Desc;
  HRESULT HResult;
  HDC BitmapDC,
      SurfaceDC;
  HBITMAP BitmapHandle;
  BITMAP Bitmap;
  LPDIRECTDRAWSURFACE4 lpDDS4Offscreen;
  LPDIRECTDRAWCLIPPER lpDDCClipper;
  TClipList ClipList;

  /* Valeur retourne par dfaut */

  *pblValid = FALSE;

  /* Initialiser l'objet */

  _hWindow = hWindow;
  _lpDD4Interface = NULL;
  _lpDDS4Backbuffer = NULL;
  _lpDDS4Primary = NULL;

  /* Crer un objet DirectDraw en utilisant le "DirectDraw device driver" par dfaut */

  HResult = DirectDrawCreate (NULL,
                              &lpDDInterface,
                              NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDraw4 */

  HResult = lpDDInterface->QueryInterface (IID_IDirectDraw4,
                                           (LPVOID *) &_lpDD4Interface);
  if (HResult != S_OK)
    return;
    
  /* Librer l'interface IDirectDraw */

  if (lpDDInterface->Release () != 0)
    return;

  /* Indiquer le niveau de coopration */

  HResult = _lpDD4Interface->SetCooperativeLevel (_hWindow,
                                                  DDSCL_EXCLUSIVE |
                                                  DDSCL_FULLSCREEN);
  if (HResult != DD_OK)
    return;

  /* Changer de mode vido */

  HResult = _lpDD4Interface->SetDisplayMode (SCREEN_DX,
                                             SCREEN_DY,
                                             16,
                                             0,
                                             0);
  if (HResult != DD_OK)
    return;

  /* Crr la surface primaire et son backbuffer */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  DDSD2Desc.dwFlags = DDSD_CAPS |
                      DDSD_BACKBUFFERCOUNT;
  DDSD2Desc.dwBackBufferCount = 1;
  DDSD2Desc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                             DDSCAPS_COMPLEX |
                             DDSCAPS_FLIP;
  HResult = _lpDD4Interface->CreateSurface (&DDSD2Desc,
                                            &_lpDDS4Primary,
                                            NULL);
  if (HResult != DD_OK)
    return;

  /* Crer une surface offscreen */

  memset (&DDSD2Desc, 0, sizeof (DDSURFACEDESC2));
  DDSD2Desc.dwSize = sizeof (DDSURFACEDESC2);
  DDSD2Desc.dwFlags = DDSD_CAPS |
                      DDSD_WIDTH |
                      DDSD_HEIGHT |
                      DDSD_PIXELFORMAT;
  DDSD2Desc.dwWidth = SCREEN_DX;
  DDSD2Desc.dwHeight = SCREEN_DY;
  DDSD2Desc.ddpfPixelFormat.dwSize = sizeof (DDPIXELFORMAT);
  DDSD2Desc.ddpfPixelFormat.dwFlags = DDPF_RGB;
  DDSD2Desc.ddpfPixelFormat.dwRGBBitCount = 16;
  DDSD2Desc.ddpfPixelFormat.dwRBitMask = 0xF800;
  DDSD2Desc.ddpfPixelFormat.dwGBitMask = 0x07E0;
  DDSD2Desc.ddpfPixelFormat.dwBBitMask = 0x001F;
  DDSD2Desc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
  HResult = _lpDD4Interface->CreateSurface (&DDSD2Desc,
                                            &lpDDS4Offscreen,
                                            NULL);
  if (HResult != DD_OK)
    return;

  /* Obtenir l'interface IDirectDrawSurface4 du backbuffer */

  DDSC2Caps.dwCaps = DDSCAPS_BACKBUFFER;
  HResult = _lpDDS4Primary->GetAttachedSurface (&DDSC2Caps,
                                                &_lpDDS4Backbuffer);
  if (HResult != DD_OK)
    return;

  /* Dcompresser les donnes du fichier BMP */

  BitmapHandle = (HBITMAP) LoadImage (NULL,
                                      BITMAP_FILE,
                                      IMAGE_BITMAP,
                                      0, 0,
                                      LR_LOADFROMFILE);
  if (BitmapHandle == NULL)
    return;
  if (GetObject (BitmapHandle, sizeof (BITMAP), &Bitmap) == 0)
    return;
  BitmapDC = CreateCompatibleDC (NULL);
  if (BitmapDC == NULL)
    return;
  SelectObject (BitmapDC, BitmapHandle);

  /* Recopier le contenu du fichier BMP dans la surface offscreen */

  HResult = lpDDS4Offscreen->GetDC (&SurfaceDC);
  if (HResult != DD_OK)
    return;
  BitBlt (SurfaceDC,
          0, 0,
          Bitmap.bmWidth,
          Bitmap.bmHeight,
          BitmapDC,
          0, 0,
          SRCCOPY);
  HResult = lpDDS4Offscreen->ReleaseDC (SurfaceDC);
  if (HResult != DD_OK)
    return;
  DeleteDC (BitmapDC);

  /* Crer un objet Clipper */
  
  HResult = _lpDD4Interface->CreateClipper (0,
                                            &lpDDCClipper,
                                            NULL);
  if (HResult != DD_OK)
    return;

  /* Associer l'objet Clipper au backbuffer */

  HResult = _lpDDS4Backbuffer->SetClipper (lpDDCClipper);
  if (HResult != DD_OK)
    return;

  /* Crer une clip list de 2 zones rectangulaires */

  ClipList._Rects[0].top = 10;
  ClipList._Rects[0].left = 10;
  ClipList._Rects[0].bottom = 200;
  ClipList._Rects[0].right = 200;
  ClipList._Rects[1].top = 30;
  ClipList._Rects[1].left = 230;
  ClipList._Rects[1].bottom = 350;
  ClipList._Rects[1].right = 400;
  ClipList._Header.dwSize = sizeof (RGNDATAHEADER);
  ClipList._Header.iType = RDH_RECTANGLES;
  ClipList._Header.nCount = 2;
  ClipList._Header.nRgnSize = ClipList._Header.nCount * sizeof (RECT);
  GetClientRect (_hWindow, &ClipList._Header.rcBound);

  HResult = lpDDCClipper->SetClipList ((LPRGNDATA) &ClipList, 0);
  if (HResult != DD_OK)
    return;

  /* Blitter le contenu de la surface offscreen dans le backbuffer */

  HResult= _lpDDS4Backbuffer->Blt (NULL,
                                   lpDDS4Offscreen,
                                   NULL,
                                   DDBLT_WAIT,
                                   NULL);
  if (HResult != DD_OK)
    return;

  /* Dtruire la surface offscreen */

  lpDDS4Offscreen->Release ();

  /* Dissocier et dtruire l'objet Clipper */

  HResult = _lpDDS4Backbuffer->SetClipper (NULL);
  if (HResult != DD_OK)
    return;
  lpDDCClipper->Release ();

  /* Valider l'initialisation */

  *pblValid = TRUE;
                                   
  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Inverser la surface primaire et son backbuffer                           */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

void TDDDemo::Flip (void)
{
  HRESULT HResult;

  /* Inverser la surface primaire et son backbuffer */

  HResult = _lpDDS4Primary->Flip (NULL,
                                  0);
  if (HResult != DD_OK)
    return;

  return;
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Destructeur                                                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   Rien                                                                     */
/*----------------------------------------------------------------------------*/

TDDDemo::~TDDDemo (void)
{
  HRESULT HResult;

  /* Restaurer le mode vido */

  HResult = _lpDD4Interface->RestoreDisplayMode ();
  if (HResult != DD_OK)
    return;

  /* Librer les interfaces */

  if (_lpDDS4Backbuffer != NULL)
    _lpDDS4Backbuffer->Release ();
  if (_lpDDS4Primary != NULL)
    _lpDDS4Primary->Release ();
  if (_lpDD4Interface != NULL)
    _lpDD4Interface->Release ();

  return;
}  
  
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Restaurer le niveau de coopration exclusif et les surfaces perdues      */
/*                                                                            */
/* ENTREE:                                                                    */
/*   Rien                                                                     */
/*                                                                            */
/* SORTIE:                                                                    */
/*   FALSE si erreur, TRUE dans le cas contraire                              */
/*----------------------------------------------------------------------------*/

void TDDDemo::Restore (void)
{
  HRESULT HResult;

  /* Attendre le retour au mode coopratif exclusif */

  while (_lpDD4Interface->TestCooperativeLevel () != DD_OK)
    WaitMessage ();

  /* Restaurer les surfaces perdues */

  HResult = _lpDD4Interface->RestoreAllSurfaces ();
  if (HResult != DD_OK)
    return;

  return;
}

/******************************************************************************/
/*                                                                            */
/*                                 Application                                */
/*                                                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Fonction de rappel de la fentre principale                              */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

LRESULT CALLBACK WindowProc (HWND hWindow,
                             UINT uMessage,
                             WPARAM wParam,
                             LPARAM lParam)
{
  switch (uMessage)
  { 
    /* Pression d'une touche */

    case WM_CHAR:

      /* Si la touche est "g", lancer la dmonstration si elle ne l'est pas dj */

      if ((wParam == 'g') && (Globals._blValid == FALSE))
      {
        Globals._pDDDemo = new TDDDemo (hWindow,
                                        Globals._hInstance,
                                        &Globals._blValid);
        if (Globals._blValid == FALSE)
          delete (Globals._pDDDemo);
        break;
      }

      /* Si la dmonstration n'est pas lance, en rester l */

      if (Globals._blValid == FALSE)
        break;

      /* Tester les touches qui permettent de contrler le droulement de la dmonstration */  

      switch (wParam)
      {
        /* 'ESC': Quitter la dmonstration et l'application */

        case VK_ESCAPE:
          Globals._blValid = FALSE;
          delete (Globals._pDDDemo);
          DestroyWindow (hWindow);
          break;

        /* 'f': Inverser la surface primaire et le backbuffer */
          
        case 'f':
          Globals._pDDDemo->Flip ();
          break;
      }
      break;

    /* Destruction de l'application */

    case WM_DESTROY:
      PostQuitMessage (0);
      break;

    /* Activation/dsactivation de l'application */

    case WM_ACTIVATEAPP:
      Globals._blActive = wParam;
      if ((Globals._blActive == FALSE) || (Globals._blValid == FALSE))
        break;
      Globals._pDDDemo->Restore ();
      break;
  }

  return (DefWindowProc (hWindow, uMessage, wParam, lParam));
}

/******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DESCRIPTION:                                                               */
/*   Point d'entre de l'application                                          */
/*                                                                            */
/* ENTREE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*                                                                            */
/* SORTIE:                                                                    */
/*   cf. le fichier d'aide de Win32                                           */
/*----------------------------------------------------------------------------*/

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int nCmdShow)
{
  HWND hWindow;
  WNDCLASS WClass;
  ATOM Atom;
  MSG Message;

  /* Initialiser les variables globales */

  Globals._hInstance = hInstance;
  Globals._pDDDemo = NULL;
  Globals._blValid = FALSE;

  /* Crer une classe de fentre */

  WClass.style = 0;
  WClass.lpfnWndProc = WindowProc;
  WClass.cbClsExtra = 0;
  WClass.cbWndExtra = 0;
  WClass.hInstance = hInstance;
  WClass.hIcon = NULL;
  WClass.hCursor = NULL;
  WClass.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
  WClass.lpszMenuName = NULL;
  WClass.lpszClassName = "DirectDraw";

  Atom = RegisterClass (&WClass);
  if (Atom == 0)
    return (0);

  /* Crer la fentre */

  hWindow = CreateWindow ("DirectDraw",
                          "Exemple DirectDraw",
                          WS_CAPTION |
                          WS_SYSMENU,
                          10,
                          50,
                          300,
                          200,
                          NULL,
                          NULL,
                          Globals._hInstance,
                          NULL);
  if (hWindow == 0)
    return (0);

  /* Montrer la fentre */

  ShowWindow (hWindow, nCmdShow);
  UpdateWindow (hWindow); 

  /* Boucle de messages */ 

  while (GetMessage (&Message, NULL, 0, 0) == TRUE)
  { 
    TranslateMessage (&Message); 
    DispatchMessage (&Message); 
  } 
  
  return (0);
}
