package com.yaps.petstore.stateless.customer;

import com.yaps.petstore.entity.Address;
import com.yaps.petstore.entity.customer.Customer;
import com.yaps.petstore.exception.ValidationException;
import com.yaps.petstore.util.Constants;

import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.List;
import java.util.logging.Logger;

/**
 * This class is a facade for all customer services.
 *
 * @author Antonio Goncalves
 */
@SuppressWarnings(value = "unchecked")
@TransactionAttribute(value = TransactionAttributeType.REQUIRED)
@Stateless(name = "CustomerSB", mappedName = "ejb/stateless/Customer")
public class CustomerBean implements CustomerRemote, CustomerLocal {

    // ======================================
    // =             Attributs              =
    // ======================================
    @PersistenceContext(unitName = "petstorePU")
    private EntityManager em;

    private final String cname = this.getClass().getName();
    private Logger logger = Logger.getLogger(Constants.LOGGER_STATELESS);

    // ======================================
    // =          Methodes publiques        =
    // ======================================
    public Customer authenticate(final String login, final String password) {
        final String mname = "authenticate";
        logger.entering(cname, mname, new Object[]{login, password});

        // On s'assure de la validit des paramtres
        if (login == null || "".equals(login))
            throw new ValidationException("Invalid login");

        Query query;
        Customer customer;

        // On recherche l'objet  partir de son login
        query = em.createQuery("SELECT c FROM Customer c WHERE c.login=:login");
        query.setParameter("login", login);
        customer = (Customer) query.getSingleResult();

        // Check if it's the right password
        if (customer != null)
            customer.matchPassword(password);

        logger.exiting(cname, mname, customer);
        return customer;
    }

    public Customer createCustomer(final Customer customer, final Address homeAddress) {
        final String mname = "createCustomer";
        logger.entering(cname, mname, customer);

        // On s'assure de la validit des paramtres
        if (customer == null)
            throw new ValidationException("Customer object is null");

        customer.setHomeAddress(homeAddress);

        em.persist(customer);

        logger.exiting(cname, mname, customer);
        return customer;
    }

    public Customer findCustomer(final Long customerId) {
        final String mname = "findCustomer";
        logger.entering(cname, mname, customerId);

        // On s'assure de la validit des paramtres
        if (customerId == null)
            throw new ValidationException("Invalid id");

        Customer customer;

        // On recherche l'objet  partir de son identifiant
        customer = em.find(Customer.class, customerId);

        logger.exiting(cname, mname, customer);
        return customer;
    }

    public void deleteCustomer(final Customer customer) {
        final String mname = "deleteCustomer";
        logger.entering(cname, mname, customer);

        // On s'assure de la validit des paramtres
        if (customer == null)
            throw new ValidationException("Customer object is null");

        // On supprime l'objet de la base de donnes
        em.remove(em.merge(customer));

        logger.exiting(cname, mname);
    }

    public Customer updateCustomer(final Customer customer, final Address homeAddress) {
        final String mname = "updateCustomer";
        logger.entering(cname, mname, customer);

        // On s'assure de la validit des paramtres
        if (customer == null)
            throw new ValidationException("Customer object is null");

        customer.setHomeAddress(homeAddress);

        // On modifie l'objet de la base de donnes
        em.merge(customer);

        logger.exiting(cname, mname, customer);
        return customer;
    }

    public List<Customer> findCustomers() {
        final String mname = "findCustomers";
        logger.entering(cname, mname);

        Query query;
        List<Customer> customers;

        // On modifie l'objet de la base de donnes
        query = em.createQuery("SELECT c FROM Customer c");
        customers = query.getResultList();

        logger.exiting(cname, mname, customers.size());
        return customers;
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================
}
