// ------------------------------------------------------------
// - La programmation orientee aspect pour Java/J2EE          -
// - Renaud Pawlak, Jean-Philippe Retaille, Lionel Seinturier -
// - Editions Eyrolles                                        -
// -                                                          -
// - Partie III  : Applications de la POA                     -
// - Chapitre 10 : Serveurs d'applications et POA             -
// ------------------------------------------------------------

package aop.as.hibernate;

import java.io.Serializable;
import java.util.Hashtable;
import java.util.List;

import net.sf.hibernate.HibernateException;
import net.sf.hibernate.MappingException;
import net.sf.hibernate.Session;
import net.sf.hibernate.SessionFactory;
import net.sf.hibernate.Transaction;
import net.sf.hibernate.cfg.Configuration;
import net.sf.hibernate.tool.hbm2ddl.SchemaExport;

/**
 * Classe utilitaire grant le framework Hibernate
 * 
 * @author Renaud Pawlak
 * @author Jean-Philippe Retaille
 * @author Lionel Seinturier
 */
public class HibernateHelper {
    private static HibernateHelper singleton = new HibernateHelper();
	private Configuration cfg = new Configuration();
	private boolean rebuildsf = true;
	private SessionFactory sf;
	private Session session;
	private Transaction transaction;
	
	private HibernateHelper() {
	}
	
    public static HibernateHelper getInstance() {
        return singleton;
    }
    
    public void addClass(Class cl) {
		try {
			cfg.addClass(cl);
			rebuildsf = true; 
		 } catch (MappingException e) {
			 throw new RuntimeException("Impossible de charger la classe "+cl.getName());
		 }
    }

    private SessionFactory getSessionFactory() throws HibernateException {
        if (rebuildsf) {
            sf = cfg.buildSessionFactory();
            rebuildsf = false;
        }
        return sf;
    }
    
    public void openSessionAndBeginTx() {
    	try {
    		SessionFactory sf = getSessionFactory();
        	session = sf.openSession();
        	transaction = session.beginTransaction();
    	}
    	catch(HibernateException e) {
			throw new RuntimeException("Ouverture de session Hibernate impossible : "+e.getMessage());
    	}
    }
    
    public void closeSessionAndCommitTx() {
		boolean commitFailed = false;
		try {
    		transaction.commit();
    	}
    	catch (HibernateException e) {
    		commitFailed = true;
    	}
    	finally {
			try {
				session.close();
			}
			catch(HibernateException e) {
				throw new RuntimeException("Erreur pendant la fermeture de la session");
			}
			if (commitFailed) {
				throw new RuntimeException("Erreur pendant le commit de la transaction de la session");
			}
    	}
    }
    
    public void closeSessionAndRollbackTx() {
		boolean rollbackFailed = false;
		try {
			transaction.rollback();
		}
		catch(HibernateException e) {
			rollbackFailed = true;
		}
		finally {
			try {
				session.close();
			}
			catch(HibernateException e) {
				throw new RuntimeException("Erreur pendant la fermeture de la session");
			}
			if (rollbackFailed) {
				throw new RuntimeException("Erreur pendant le rollback de la transaction de la session");
			}
		}
		
    }
    
    public Session getSession() {
        if ( session == null || !session.isOpen() ) {
			throw new RuntimeException("openSessionAndBeginTx doit tre appele avant");
        }
        return session;
    }
    
    public void load(Object o,Object primaryKey) {
    	try {
    		session.load(o,(Serializable)primaryKey);	
    	}
    	catch(HibernateException e) {
			closeSessionAndRollbackTx();
			throw new RuntimeException("Chargement de l'objet "+o.getClass().getName()+" impossible avec la cl primaire "+primaryKey);
    	}
    }
    
	public void save(Object o) {
		try {
			session.save(o);
		}
		catch(HibernateException he){
			closeSessionAndRollbackTx();
			throw new RuntimeException("Cration de l'objet "+o.getClass().getName()+" impossible");
		}
	}
	
	public void delete(Object o) {
    	try {
    		session.delete(o);
    	}
    	catch(HibernateException he) {
			closeSessionAndRollbackTx();
			throw new RuntimeException("Suppression de l'objet "+o+" impossible");
    	}
	}
}