package com.yaps.petstore.mdb.print;

import com.yaps.petstore.entity.order.Order;
import com.yaps.petstore.entity.order.OrderLine;
import com.yaps.petstore.util.Constants;

import javax.ejb.EJBException;
import javax.ejb.MessageDriven;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import java.text.SimpleDateFormat;
import java.util.logging.Logger;

/**
 * La classe OrderPrinterBean est un Message Driven bean.
 * Elle implmente l'interface javax.jms.MessageListener.
 * <p/>
 * Ce MDB se connecte de faon durable au topic ddi aux commandes,
 * et, en cas d'achat d'un chien dangereux, l'animal est automatiquement dclar au prs des autorits.
 * Pour ce faire un message est post sur une queue JMS  laquelle se connecte l'administration.
 *
 * @author Alexis Midon
 * @see javax.jms.MessageListener
 */
@MessageDriven(mappedName = "jms/topic/order")
public class OrderPrinterBean implements MessageListener {

    // ======================================
    // =             Attributs              =
    // ======================================
//    @Resource
//    private MessageDrivenContext mdc;
//    @Resource(mappedName = "jms/petstoreConnectionFactory")
//    private ConnectionFactory connectionFactory;
//    @Resource(mappedName = "jms/queue/legalDpt")
//    private Destination destinationOrder;
//
//    private Connection connection;

    private final String cname = this.getClass().getName();
    private Logger logger = Logger.getLogger(Constants.LOGGER_MDB);

    private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

    // ======================================
    // =             Constantes             =
    // ======================================

    // ======================================
    // =            Constructeurs           =
    // ======================================

    // ======================================
    // =          Methodes publiques        =
    // ======================================

    public void onMessage(Message message) {
        final String mname = "onMessage";
        logger.entering(cname, mname);
        try {
            if (message instanceof ObjectMessage) {
                ObjectMessage msg = (ObjectMessage) message;
                logger.info("Message received! Id=" + msg.getJMSMessageID());
                Order order = (Order) msg.getObject();
                printOrder(order);
            } else {
                logger.warning("Message of wrong type: Id=" + message.getJMSMessageID() + " class=" + message.getClass().getName());
            }
        } catch (JMSException e) {
            logger.throwing(cname, mname, e);
            // rollback the undergoing transaction if error occured
            throw new EJBException(e);
        }
        logger.exiting(cname, mname);
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================

    private void printOrder(Order order) {
        logger.info("Order # " + order.getId() + " on the " + dateFormat.format(order.getOrderDate()));
        logger.info(order.getCustomer().getFirstname() + order.getCustomer().getLastname() + " bought ");

        for (OrderLine line : order.getOrderLines()) {
            logger.info("\t" + line.getItem().getName() + "*" + line.getQuantity() + "=" + line.getSubTotal());
        }
        logger.info("Total=" + order.getTotal());
    }
}
