package com.yaps.petstore.stateless.catalog;

import com.yaps.petstore.entity.catalog.Category;
import com.yaps.petstore.entity.catalog.Item;
import com.yaps.petstore.entity.catalog.Product;
import com.yaps.petstore.exception.ValidationException;
import com.yaps.petstore.util.Constants;

import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.List;
import java.util.logging.Logger;

/**
 * This class is a facade for all catalog services.
 *
 * @author Antonio Goncalves
 */
@SuppressWarnings(value = "unchecked")
@TransactionAttribute(value = TransactionAttributeType.REQUIRED)
@Stateless(name = "CatalogSB", mappedName = "ejb/stateless/Catalog")
public class CatalogBean implements CatalogRemote, CatalogLocal {

    // ======================================
    // =             Attributs              =
    // ======================================
    @PersistenceContext(unitName = "petstorePU")
    private EntityManager em;

    private final String cname = this.getClass().getName();
    private Logger logger = Logger.getLogger(Constants.LOGGER_STATELESS);

    // ======================================
    // = Methodes publiques pour categorie  =
    // ======================================
    public Category createCategory(final Category category) {
        final String mname = "getNewCategory";
        logger.entering(cname, mname, category);

        // On s'assure de la validit des paramtres
        if (category == null)
            throw new ValidationException("Category object is null");

        // L'objet est persist en base de donnes
        em.persist(category);

        logger.exiting(cname, mname, category);
        return category;
    }

    public Category findCategory(final Long categoryId) {
        final String mname = "findCategory";
        logger.entering(cname, mname, categoryId);

        // On s'assure de la validit des paramtres
        if (categoryId == null)
            throw new ValidationException("Invalid id");

        Category category;

        // On recherche l'objet  partir de son identifiant
        category = em.find(Category.class, categoryId);

        logger.exiting(cname, mname, category);
        return category;
    }

    public void deleteCategory(final Category category) {
        final String mname = "deleteCategory";
        logger.entering(cname, mname, category);

        // On s'assure de la validit des paramtres
        if (category == null)
            throw new ValidationException("Category object is null");

        // On supprime l'objet de la base de donnes
        em.remove(em.merge(category));

        logger.exiting(cname, mname);
    }

    public Category updateCategory(final Category category) {
        final String mname = "updateCategory";
        logger.entering(cname, mname, category);

        // On s'assure de la validit des paramtres
        if (category == null)
            throw new ValidationException("Category object is null");

        // On modifie l'objet de la base de donnes
        em.merge(category);

        logger.exiting(cname, mname, category);
        return category;
    }

    public List<Category> findCategories() {
        final String mname = "findCategories";
        logger.entering(cname, mname);

        Query query;
        List<Category> categories;

        // On modifie l'objet de la base de donnes
        query = em.createQuery("SELECT c FROM Category c ORDER BY c.name");
        categories = query.getResultList();

        logger.exiting(cname, mname, categories.size());
        return categories;
    }

    // ======================================
    // = Methodes publiques pour produit    =
    // ======================================
    public Product createProduct(final Product product, final Category category) {
        final String mname = "createProduct";
        logger.entering(cname, mname, product);

        // On s'assure de la validit des paramtres
        if (product == null)
            throw new ValidationException("Product object is null");
        if (category == null)
            throw new ValidationException("Product must be attached to a category");

        product.setCategory(category);

        // L'objet est persist en base de donnes
        em.persist(product);

        logger.exiting(cname, mname, product);
        return product;
    }

    public Product findProduct(final Long productId) {
        final String mname = "findProduct";
        logger.entering(cname, mname, productId);

        // On s'assure de la validit des paramtres
        if (productId == null)
            throw new ValidationException("Invalid id");

        Product product;

        // On recherche l'objet  partir de son identifiant
        product = em.find(Product.class, productId);

        logger.exiting(cname, mname, product);
        return product;
    }

    public void deleteProduct(final Product product) {
        final String mname = "deleteProduct";
        logger.entering(cname, mname, product);

        // On s'assure de la validit des paramtres
        if (product == null)
            throw new ValidationException("Product object is null");

        // On supprime l'objet de la base de donnes
        em.remove(em.merge(product));

        logger.exiting(cname, mname);
    }

    public Product updateProduct(final Product product, final Category category) {
        final String mname = "updateProduct";
        logger.entering(cname, mname, product);

        // On s'assure de la validit des paramtres
        if (product == null)
            throw new ValidationException("Product object is null");
        if (category == null)
            throw new ValidationException("Product must be attached to a category");

        product.setCategory(category);

        // On modifie l'objet de la base de donnes
        em.merge(product);

        logger.exiting(cname, mname, product);
        return product;
    }

    public List<Product> findProducts() {
        final String mname = "findCategories";
        logger.entering(cname, mname);

        Query query;
        List<Product> products;

        // On modifie l'objet de la base de donnes
        query = em.createQuery("SELECT p FROM Product p ORDER BY p.name");
        products = query.getResultList();

        logger.exiting(cname, mname, products.size());
        return products;
    }

    // ======================================
    // =   Methodes publiques pour article  =
    // ======================================
    public Item createItem(final Item item, final Product product) {
        final String mname = "createItem";
        logger.entering(cname, mname, item);

        // On s'assure de la validit des paramtres
        if (item == null)
            throw new ValidationException("Item object is null");
        if (product == null)
            throw new ValidationException("Item must be attached to a product");

        item.setProduct(product);

        // L'objet est persist en base de donnes
        em.persist(item);

        logger.exiting(cname, mname, item);
        return item;
    }

    public Item findItem(final Long itemId) {
        final String mname = "findItem";
        logger.entering(cname, mname, itemId);

        // On s'assure de la validit des paramtres
        if (itemId == null)
            throw new ValidationException("Invalid id");

        Item item;

        // On recherche l'objet  partir de son identifiant
        item = em.find(Item.class, itemId);


        logger.exiting(cname, mname, item);
        return item;
    }

    public void deleteItem(final Item item) {
        final String mname = "deleteItem";
        logger.entering(cname, mname, item);

        // On s'assure de la validit des paramtres
        if (item == null)
            throw new ValidationException("Item object is null");

        // On supprime l'objet de la base de donnes
        em.remove(em.merge(item));

        logger.exiting(cname, mname);
    }

    public Item updateItem(final Item item, final Product product) {
        final String mname = "updateItem";
        logger.entering(cname, mname, item);

        // On s'assure de la validit des paramtres
        if (item == null)
            throw new ValidationException("Item object is null");
        if (product == null)
            throw new ValidationException("Item must be attached to a product");

        item.setProduct(product);

        // On modifie l'objet de la base de donnes
        em.merge(item);

        logger.exiting(cname, mname, item);
        return item;
    }

    public List<Item> findItems() {
        final String mname = "findCategories";
        logger.entering(cname, mname);

        Query query;
        List<Item> items;

        // On modifie l'objet de la base de donnes
        query = em.createQuery("SELECT i FROM Item i ORDER BY i.name");
        items = query.getResultList();

        logger.exiting(cname, mname, items.size());
        return items;
    }


    public List<Item> searchItems(final String keyword) {
        final String mname = "searchItems";
        logger.entering(cname, mname, keyword);

        Query query;
        List<Item> items;

        // On modifie l'objet de la base de donnes
        query = em.createQuery("SELECT i FROM Item i WHERE UPPER(i.name) LIKE :keyword OR UPPER(i.product.name) LIKE :keyword ORDER BY i.product.category.name, i.product.name");
        query.setParameter("keyword", "%" + keyword.toUpperCase() + "%");
        items = query.getResultList();

        logger.exiting(cname, mname, items.size());
        return items;
    }
    // ======================================
    // =           Methodes Prives         =
    // ======================================
}
