package com.yaps.petstore.entity.catalog;

import static com.yaps.petstore.TestHelper.*;

import com.yaps.petstore.exception.ValidationException;
import com.yaps.petstore.util.Constants;
import org.junit.AfterClass;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import javax.persistence.*;
import java.util.List;

/**
 * Test unitaire de la classe Category
 *
 * @author Antonio Goncalves
 * @see Category
 */
public class CategoryTest {

    // ======================================
    // =             Attributs              =
    // ======================================
    private static EntityManagerFactory emf;
    private static EntityManager em;
    private static EntityTransaction tx;

    // ======================================
    // =              Fixture               =
    // ======================================

    @BeforeClass
    public static void initEntityManager() {
        emf = Persistence.createEntityManagerFactory(Constants.PERSISTENCE_UNIT_NAME);
        em = emf.createEntityManager();
    }

    @AfterClass
    public static void closeEntityManager() {
        em.close();
        emf.close();
    }

    @Before
    public void initEntityTransaction() {
        tx = em.getTransaction();
    }

    // ======================================
    // =              Test cases            =
    // ======================================

    /**
     * Ce test s'assure que la cration, lecture, mise  jour et suppression d'un objet
     * en base de donnes fonctionne.
     */
    @Test
    public void crud() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Category category = new Category();

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAll();

        // Cre un objet et le persiste en base de donne
        category = getMockCategoryValues(category, random);
        tx.begin();
        em.persist(category);
        tx.commit();
        Long id = category.getId();

        // On s'assure que l'objet a bien t cr et que les valeurs sont bien les bonnes
        category = em.find(Category.class, id);
        assertNotNull("Object should exist", category);
        checkMockCategoryValues(category, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        category = getMockCategoryValues(category, updateRandom);
        tx.begin();
        em.merge(category);
        tx.commit();

        // On s'assure que l'objet a bien t mis  jour et que les valeurs sont bien les bonnes
        category = em.find(Category.class, id);
        assertNotNull("Object should exist", category);
        checkMockCategoryValues(category, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAll();

        // ...et on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        tx.begin();
        em.remove(category);
        tx.commit();

        // On s'assure que l'objet a bien t supprim
        category = em.find(Category.class, id);
        assertNull("Object should not exist", category);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAll();

        // ...et on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * Ce test s'assure que la cration et la modification prennent en compte les
     * valeurs obligatoires.
     */
    @Test
    public void invalidValues() throws Exception {
        Category category = null;

        // Essaie de crer un objet avec des valeurs vides
        try {
            category = new Category("", "");
            tx.begin();
            em.persist(category);
            fail("Object with empty values should not be created");
        } catch (ValidationException e) {
            tx.rollback();
        }

        // Essaie de crer un objet avec des valeurs null
        try {
            category = new Category(null, null);
            tx.begin();
            em.persist(category);
            fail("Object with null values should not be created");
        } catch (ValidationException e) {
            tx.rollback();
        }

        // Cre un objet et le persiste en base de donne
        category = getMockCategoryValues(category, getRandom());
        tx.begin();
        em.persist(category);
        tx.commit();
        Long id = category.getId();

        // Essaie de modifier l'objet avec des valeurs vides
        try {
            category.setName("");
            category.setDescription("");
            tx.begin();
            em.merge(category);
            tx.commit();
            fail("Object with empty values should not be updated");
        } catch (RollbackException e) {
        }

        // Essaie de modifier l'objet avec des valeurs null
        try {
            category.setName(null);
            category.setDescription(null);
            tx.begin();
            em.merge(category);
            tx.commit();
            fail("Object with null values should not be updated");
        } catch (RollbackException e) {
        }

        // On supprime l'objet de la base de donnes
        tx.begin();
        em.remove(em.find(Category.class, id));
        tx.commit();
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================
    private int findAll() {
        Query query = em.createQuery("SELECT a FROM Category a");
        List result = query.getResultList();
        if (result == null)
            return 0;
        else
            return result.size();
    }
}
