package com.yaps.petstore.entity.order;

import static com.yaps.petstore.TestHelper.*;
import com.yaps.petstore.entity.catalog.Category;
import com.yaps.petstore.entity.catalog.Item;
import com.yaps.petstore.entity.catalog.Product;
import com.yaps.petstore.util.Constants;
import org.junit.AfterClass;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import javax.persistence.*;
import java.util.List;

/**
 * Test unitaire de la classe OrderLine
 *
 * @author Antonio Goncalves
 * @see com.yaps.petstore.entity.Address
 */
public class OrderLineTest {

    // ======================================
    // =             Attributs              =
    // ======================================
    private static EntityManagerFactory emf;
    private static EntityManager em;
    private static EntityTransaction tx;

    // ======================================
    // =              Fixture               =
    // ======================================

    @BeforeClass
    public static void initEntityManager() {
        emf = Persistence.createEntityManagerFactory(Constants.PERSISTENCE_UNIT_NAME);
        em = emf.createEntityManager();
    }

    @AfterClass
    public static void closeEntityManager() {
        em.close();
        emf.close();
    }

    @Before
    public void initEntityTransaction() {
        tx = em.getTransaction();
    }

    // ======================================
    // =              Test cases            =
    // ======================================

    /**
     * Ce test s'assure que la cration, lecture, mise  jour et suppression d'un objet
     * en base de donnes fonctionne.
     */
    @Test
    public void crud() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        OrderLine orderLine = new OrderLine();
        Item item = new Item();
        Product product = new Product();
        Category category = new Category();

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAll();

        // Cre un objet et le persiste en base de donne
        category = getMockCategoryValues(category, random);
        product = getMockProductValues(product, random);
        product.setCategory(category);
        item = getMockItemValues(item, random);
        item.setProduct(product);

        orderLine = getMockOrderLineValues(orderLine, random);
        orderLine.setItem(item);

        tx.begin();
        em.persist(category);
        em.persist(product);
        em.persist(item);
        em.persist(orderLine);
        tx.commit();
        Long id = orderLine.getId();

        // On s'assure que l'objet a bien t cr et que les valeurs sont bien les bonnes
        orderLine = em.find(OrderLine.class, id);
        assertNotNull("Object should exist", orderLine);
        checkMockOrderLineValues(orderLine, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        orderLine = getMockOrderLineValues(orderLine, updateRandom);
        tx.begin();
        em.merge(orderLine);
        tx.commit();

        // On s'assure que l'objet a bien t mis  jour et que les valeurs sont bien les bonnes
        orderLine = em.find(OrderLine.class, id);
        assertNotNull("Object should exist", orderLine);
        checkMockOrderLineValues(orderLine, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAll();

        // ...et on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        item = orderLine.getItem();
        tx.begin();
        em.remove(orderLine);
        em.remove(item);
        em.remove(product);
        em.remove(category);
        tx.commit();

        // On s'assure que l'objet a bien t supprim
        orderLine = em.find(OrderLine.class, id);
        assertNull("Object should not exist", orderLine);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAll();

        // ...et on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================
    private int findAll() {
        Query query = em.createQuery("SELECT o FROM OrderLine o");
        List result = query.getResultList();
        if (result == null)
            return 0;
        else
            return result.size();
    }
}