package com.yaps.petstore.entity.order;

import static com.yaps.petstore.TestHelper.*;

import com.yaps.petstore.entity.customer.Customer;
import com.yaps.petstore.util.Constants;
import org.junit.AfterClass;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import javax.persistence.*;
import java.util.List;

/**
 * Test unitaire de la classe Order
 *
 * @author Antonio Goncalves
 * @see com.yaps.petstore.entity.Address
 */
public class OrderTest {

    // ======================================
    // =             Attributs              =
    // ======================================
    private static EntityManagerFactory emf;
    private static EntityManager em;
    private static EntityTransaction tx;

    // ======================================
    // =              Fixture               =
    // ======================================

    @BeforeClass
    public static void initEntityManager() {
        emf = Persistence.createEntityManagerFactory(Constants.PERSISTENCE_UNIT_NAME);
        em = emf.createEntityManager();
    }

    @AfterClass
    public static void closeEntityManager() {
        em.close();
        emf.close();
    }

    @Before
    public void initEntityTransaction() {
        tx = em.getTransaction();
    }

    // ======================================
    // =              Test cases            =
    // ======================================

    /**
     * Ce test s'assure que la cration, lecture, mise  jour et suppression d'un objet
     * en base de donnes fonctionne.
     */
    @Test
    public void crud() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Order order = new Order();
        Customer customer = new Customer();

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAll();

        // Cre un objet et le persiste en base de donne
        customer = getMockCustomerValues(customer, random);
        order = getMockOrderValues(order, random);
        order.setCustomer(customer);
        tx.begin();
        em.persist(order);
        em.persist(customer);
        tx.commit();
        Long id = order.getId();

        // On s'assure que l'objet a bien t cr et que les valeurs sont bien les bonnes
        order = em.find(Order.class, id);
        assertNotNull("Object should exist", order);
        checkMockOrderValues(order, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        order = getMockOrderValues(order, updateRandom);
        tx.begin();
        em.merge(order);
        tx.commit();

        // On s'assure que l'objet a bien t mis  jour et que les valeurs sont bien les bonnes
        order = em.find(Order.class, id);
        assertNotNull("Object should exist", order);
        checkMockOrderValues(order, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAll();

        // ...et on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        tx.begin();
        em.remove(order);
        tx.commit();

        // On s'assure que l'objet a bien t supprim
        order = em.find(Order.class, id);
        assertNull("Object should not exist", order);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAll();

        // ...et on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================
    private int findAll() {
        Query query = em.createQuery("SELECT a FROM Order a");
        List result = query.getResultList();
        if (result == null)
            return 0;
        else
            return result.size();
    }
}