package com.yaps.petstore.entity;

import static com.yaps.petstore.TestHelper.*;
import com.yaps.petstore.exception.ValidationException;
import com.yaps.petstore.util.Constants;
import org.junit.AfterClass;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import javax.persistence.*;
import java.util.List;

/**
 * Test unitaire de la classe Address
 *
 * @author Antonio Goncalves
 * @see Address
 */
public class AddressTest {

    // ======================================
    // =             Attributs              =
    // ======================================
    private static EntityManagerFactory emf;
    private static EntityManager em;
    private static EntityTransaction tx;

    // ======================================
    // =              Fixture               =
    // ======================================

    @BeforeClass
    public static void initEntityManager() {
        emf = Persistence.createEntityManagerFactory(Constants.PERSISTENCE_UNIT_NAME);
        em = emf.createEntityManager();
    }

    @AfterClass
    public static void closeEntityManager() {
        em.close();
        emf.close();
    }

    @Before
    public void initEntityTransaction() {
        tx = em.getTransaction();
    }

    // ======================================
    // =              Test cases            =
    // ======================================

    /**
     * Ce test s'assure que la cration, lecture, mise  jour et suppression d'un objet
     * en base de donnes fonctionne.
     */
    @Test
    public void crud() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Address address = new Address();

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAll();

        // Cre un objet et le persiste en base de donne
        address = getMockAddressValues(address, random);
        tx.begin();
        em.persist(address);
        tx.commit();
        Long id = address.getId();

        // On s'assure que l'objet a bien t cr et que les valeurs sont bien les bonnes
        address = em.find(Address.class, id);
        assertNotNull("Object should exist", address);
        checkMockAddressValues(address, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        address = getMockAddressValues(address, updateRandom);
        tx.begin();
        em.merge(address);
        tx.commit();

        // On s'assure que l'objet a bien t mis  jour et que les valeurs sont bien les bonnes
        address = em.find(Address.class, id);
        assertNotNull("Object should exist", address);
        checkMockAddressValues(address, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAll();

        // ...et on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        tx.begin();
        em.remove(address);
        tx.commit();

        // On s'assure que l'objet a bien t supprim
        address = em.find(Address.class, id);
        assertNull("Object should not exist", address);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAll();

        // ...et on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * Ce test s'assure que la cration et la modification prennent en compte les
     * valeurs obligatoires.
     */
    @Test
    public void invalidValues() throws Exception {
        Address address;

        // Essaie de crer un objet avec des valeurs vides
        try {
            address = new Address("", "", "", "");
            tx.begin();
            em.persist(address);
            fail("Object with empty values should not be created");
        } catch (ValidationException e) {
            tx.rollback();
        }

        // Essaie de crer un objet avec des valeurs null
        try {
            address = new Address(null, null, null, null);
            tx.begin();
            em.persist(address);
            fail("Object with null values should not be created");
        } catch (ValidationException e) {
            tx.rollback();
        }
    }

    // ======================================
    // =           Methodes Prives         =
    // ======================================
    private int findAll() {
        Query query = em.createQuery("SELECT a FROM Address a");
        List result = query.getResultList();
        if (result == null)
            return 0;
        else
            return result.size();
    }
}
