package com.yaps.petstore.stateless;

import static com.yaps.petstore.TestHelper.*;

import com.yaps.petstore.delegate.CatalogDelegate;
import com.yaps.petstore.entity.catalog.Category;
import com.yaps.petstore.entity.catalog.Item;
import com.yaps.petstore.entity.catalog.Product;
import static org.junit.Assert.*;
import org.junit.Test;

import javax.ejb.EJBException;

/**
 * Test unitaire du composant Catalog
 *
 * @author Antonio Goncalves
 */
public class CatalogServiceTest {

    // ======================================
    // =      Test cases pour Category      =
    // ======================================
    /**
     * This test ensures that the method findAll works. It does a first findAll, creates
     * a new object and does a second findAll.
     */

    @Test
    public void crudCategory() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Category category;

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAllCategories();

        // Cre un objet tx le persiste en base de donne
        category = createCategory(random);
        Long id = category.getId();

        // On s'assure que l'objet  bien t cr tx que les valeurs sont bien les bonnes
        category = CatalogDelegate.findCategory(id);
        assertNotNull("Object should exist", category);
        checkMockCategoryValues(category, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        updateCategory(category, updateRandom);

        // On s'assure que l'objet  bien t mis  jour tx que les valeurs sont bien les bonnes
        category = CatalogDelegate.findCategory(id);
        assertNotNull("Object should exist", category);
        checkMockCategoryValues(category, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAllCategories();

        // ...tx on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        CatalogDelegate.deleteCategory(category);

        // On s'assure que l'objet  bien t supprim
        category = CatalogDelegate.findCategory(id);
        assertNull("Object should not exist", category);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAllCategories();

        // ...tx on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * This test tries to find an object with a invalid identifier.
     */
    @Test
    public void findCategoryWithInvalidValues() throws Exception {

        // Finds an object with a unknown identifier
        Category category = CatalogDelegate.findCategory(getRandom());
        assertNull("Object with unknonw id should not be found", category);

        // Finds an object with a null identifier
        try {
            CatalogDelegate.findCategory(null);
            fail("Object with null id should not be found");
        } catch (EJBException e) {
        }
    }


    /**
     * This test tries to create an object with a invalid values.
     */
    @Test
    public void createCategoryWithInvalidValues() throws Exception {

        Category category;

        // Creates an object with a null parameter
        try {
            CatalogDelegate.createCategory(null);
            fail("Object with null parameter should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with empty values
        try {
            category = new Category(new String(), new String());
            CatalogDelegate.createCategory(category);
            fail("Object with empty values should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with null values
        try {
            category = new Category(null, null);
            CatalogDelegate.createCategory(category);
            fail("Object with null values should not be created");
        } catch (EJBException e) {
        }
    }

    /**
     * This test tries to update an object with a invalid values.
     */
    @Test
    public void updateCategoryWithInvalidValues() throws Exception {
        Category category;

        // Updates an object with a null parameter
        try {
            CatalogDelegate.updateCategory(null);
            fail("Object with null parameter should not be updated");
        } catch (EJBException e) {
        }

        // Updates an object with null values
        try {
            category = new Category(null, null);
            CatalogDelegate.updateCategory(category);
            fail("Object with null values should not be updated");
        } catch (EJBException e) {
        }
    }

    /**
     * This test ensures that the system cannont remove an unknown object
     */
    @Test
    public void deleteUnknownCategory() throws Exception {
        Category category;

        // Removes an object with a null parameter
        try {
            CatalogDelegate.deleteCategory(null);
            fail("Object with null parameter should not be removed");
        } catch (EJBException e) {
        }
    }

    // ======================================
    // =      Test cases pour Product       =
    // ======================================
    /**
     * This test ensures that the method findAll works. It does a first findAll, creates
     * a new object and does a second findAll.
     */

    @Test
    public void crudProduct() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Product product;

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAllProducts();

        // Cre un objet tx le persiste en base de donne
        product = createProduct(random);
        Long id = product.getId();

        // On s'assure que l'objet  bien t cr tx que les valeurs sont bien les bonnes
        product = CatalogDelegate.findProduct(id);
        assertNotNull("Object should exist", product);
        checkMockProductValues(product, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        updateProduct(product, updateRandom);

        // On s'assure que l'objet  bien t mis  jour tx que les valeurs sont bien les bonnes
        product = CatalogDelegate.findProduct(id);
        assertNotNull("Object should exist", product);
        checkMockProductValues(product, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAllProducts();

        // ...tx on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        CatalogDelegate.deleteProduct(product);

        // On s'assure que l'objet  bien t supprim
        product = CatalogDelegate.findProduct(id);
        assertNull("Object should not exist", product);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAllProducts();

        // ...tx on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * This test tries to find an object with a invalid identifier.
     */
    @Test
    public void findProductWithInvalidValues() throws Exception {

        // Finds an object with a unknown identifier
        Product product = CatalogDelegate.findProduct(getRandom());
        assertNull("Object with unknonw id should not be found", product);

        // Finds an object with a null identifier
        try {
            CatalogDelegate.findProduct(null);
            fail("Object with null id should not be found");
        } catch (EJBException e) {
        }
    }


    /**
     * This test tries to create an object with a invalid values.
     */
    @Test
    public void createProductWithInvalidValues() throws Exception {

        Product product;

        // Creates an object with a null parameter
        try {
            CatalogDelegate.createProduct(null, null);
            fail("Object with null parameter should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with empty values
        try {
            product = new Product(new String(), new String());
            CatalogDelegate.createProduct(product, null);
            fail("Object with empty values should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with null values
        try {
            product = new Product(null, null);
            CatalogDelegate.createProduct(product, null);
            fail("Object with null values should not be created");
        } catch (EJBException e) {
        }
    }

    /**
     * This test tries to update an object with a invalid values.
     */
    @Test
    public void updateProductWithInvalidValues() throws Exception {
        Product product;

        // Updates an object with a null parameter
        try {
            CatalogDelegate.updateProduct(null, null);
            fail("Object with null parameter should not be updated");
        } catch (EJBException e) {
        }

        // Updates an object with null values
        try {
            product = new Product(null, null);
            CatalogDelegate.updateProduct(product, null);
            fail("Object with null values should not be updated");
        } catch (EJBException e) {
        }
    }

    /**
     * This test ensures that the system cannont remove an unknown object
     */
    @Test
    public void deleteUnknownProduct() throws Exception {
        Product product;

        // Removes an object with a null parameter
        try {
            CatalogDelegate.deleteProduct(null);
            fail("Object with null parameter should not be removed");
        } catch (EJBException e) {
        }
    }

    // ======================================
    // =        Test cases pour Item        =
    // ======================================
    /**
     * This test ensures that the method findAll works. It does a first findAll, creates
     * a new object and does a second findAll.
     */

    @Test
    public void crudItem() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Item item;

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAllItems();

        // Cre un objet tx le persiste en base de donne
        item = createItem(random);
        Long id = item.getId();

        // On s'assure que l'objet  bien t cr tx que les valeurs sont bien les bonnes
        item = CatalogDelegate.findItem(id);
        assertNotNull("Object should exist", item);
        checkMockItemValues(item, random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        updateItem(item, updateRandom);

        // On s'assure que l'objet  bien t mis  jour tx que les valeurs sont bien les bonnes
        item = CatalogDelegate.findItem(id);
        assertNotNull("Object should exist", item);
        checkMockItemValues(item, updateRandom);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAllItems();

        // ...tx on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        CatalogDelegate.deleteItem(item);

        // On s'assure que l'objet  bien t supprim
        item = CatalogDelegate.findItem(id);
        assertNull("Object should not exist", item);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAllItems();

        // ...tx on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * This test tries to find an object with a invalid identifier.
     */
    @Test
    public void findItemWithInvalidValues() throws Exception {

        // Finds an object with a unknown identifier
        Item item = CatalogDelegate.findItem(getRandom());
        assertNull("Object with unknonw id should not be found", item);

        // Finds an object with a null identifier
        try {
            CatalogDelegate.findItem(null);
            fail("Object with null id should not be found");
        } catch (EJBException e) {
        }
    }


    /**
     * This test tries to create an object with a invalid values.
     */
    @Test
    public void createItemWithInvalidValues() throws Exception {

        Item item;

        // Creates an object with a null parameter
        try {
            CatalogDelegate.createItem(null, null);
            fail("Object with null parameter should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with empty values
        try {
            item = new Item(new String(), new Float(0));
            CatalogDelegate.createItem(item, null);
            fail("Object with empty values should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with null values
        try {
            item = new Item(null, null);
            CatalogDelegate.createItem(item, null);
            fail("Object with null values should not be created");
        } catch (EJBException e) {
        }
    }

    /**
     * This test tries to update an object with a invalid values.
     */
    @Test
    public void updateItemWithInvalidValues() throws Exception {
        Item item;

        // Updates an object with a null parameter
        try {
            CatalogDelegate.updateItem(null, null);
            fail("Object with null parameter should not be updated");
        } catch (EJBException e) {
        }

        // Updates an object with null values
        try {
            item = new Item(null, null);
            CatalogDelegate.updateItem(item, null);
            fail("Object with null values should not be updated");
        } catch (EJBException e) {
        }
    }

    /**
     * This test ensures that the system cannont remove an unknown object
     */
    @Test
    public void deleteUnknownItem() throws Exception {
        Item item;

        // Removes an object with a null parameter
        try {
            CatalogDelegate.deleteItem(null);
            fail("Object with null parameter should not be removed");
        } catch (EJBException e) {
        }
    }
}