package com.yaps.petstore.stateless;

import static com.yaps.petstore.TestHelper.*;

import com.yaps.petstore.delegate.CustomerDelegate;
import com.yaps.petstore.entity.customer.Customer;
import static org.junit.Assert.*;
import org.junit.Test;

import javax.ejb.EJBException;

/**
 * Test unitaire du composant Customer
 *
 * @author Antonio Goncalves
 * @see com.yaps.petstore.entity.Address
 */
public class CustomerServiceTest {

    // ======================================
    // =              Test cases            =
    // ======================================

    /**
     * This test ensures that the method findAll works. It does a first findAll, creates
     * a new object and does a second findAll.
     */

    @Test
    public void crudCustomer() throws Exception {
        Long random = getRandom();
        Long updateRandom = getRandom();
        Customer customer;

        // Rcupre tous les objets prsent en base de donnes
        int firstFindAll = findAllCustomers();

        // Cre un objet tx le persiste en base de donne
        customer = createCustomer(random);
        Long id = customer.getId();

        // On s'assure que l'objet  bien t cr tx que les valeurs sont bien les bonnes
        customer = CustomerDelegate.findCustomer(id);
        assertNotNull("Object should exist", customer);
        checkMockCustomerValues(customer, random);
        checkMockAddressValues(customer.getHomeAddress(), random);

        // Mise  jour de l'objet avec de nouvelles valeurs
        updateCustomer(customer, updateRandom);

        // On s'assure que l'objet  bien t mis  jour tx que les valeurs sont bien les bonnes
        customer = CustomerDelegate.findCustomer(id);
        assertNotNull("Object should exist", customer);
        checkMockCustomerValues(customer, updateRandom);
        checkMockAddressValues(customer.getHomeAddress(), random);

        // On rcupre tous les objets prsent en base de donnes...
        int secondFindAll = findAllCustomers();

        // ...tx on s'assure qu'il y a une occurence supplmentaire
        if (firstFindAll + 1 != secondFindAll) fail("The collection size should have increased by 1");

        // On supprime l'objet de la base de donnes
        CustomerDelegate.deleteCustomer(customer);

        // On s'assure que l'objet  bien t supprim
        customer = CustomerDelegate.findCustomer(id);
        assertNull("Object should not exist", customer);

        // On rcupre tous les objets prsent en base de donnes...
        int thirdFindAll = findAllCustomers();

        // ...tx on s'assure qu'il y ait le mme nombre initial d'occurences
        if (firstFindAll != thirdFindAll) fail("The collection size should have be the same as original");
    }

    /**
     * This test tries to find an object with a invalid identifier.
     */
    @Test
    public void findCustomerWithInvalidValues() throws Exception {

        // Finds an object with a unknown identifier
        Customer customer = CustomerDelegate.findCustomer(getRandom());
        assertNull("Object with unknonw id should not be found", customer);

        // Finds an object with a null identifier
        try {
            CustomerDelegate.findCustomer(null);
            fail("Object with null id should not be found");
        } catch (EJBException e) {
        }
    }


    /**
     * This test tries to create an object with a invalid values.
     */
    @Test
    public void createCustomerWithInvalidValues() throws Exception {

        Customer customer;

        // Creates an object with a null parameter
        try {
            CustomerDelegate.createCustomer(null, null);
            fail("Object with null parameter should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with empty values
        try {
            customer = new Customer(new String(), new String());
            CustomerDelegate.createCustomer(customer, null);
            fail("Object with empty values should not be created");
        } catch (EJBException e) {
        }

        // Creates an object with null values
        try {
            customer = new Customer(null, null);
            CustomerDelegate.createCustomer(customer, null);
            fail("Object with null values should not be created");
        } catch (EJBException e) {
        }
    }

    /**
     * This test tries to update an object with a invalid values.
     */
    @Test
    public void updateCustomerWithInvalidValues() throws Exception {
        Customer customer;

        // Updates an object with a null parameter
        try {
            CustomerDelegate.updateCustomer(null, null);
            fail("Object with null parameter should not be updated");
        } catch (EJBException e) {
        }

        // Updates an object with null values
        try {
            customer = new Customer(null, null);
            CustomerDelegate.updateCustomer(customer, null);
            fail("Object with null values should not be updated");
        } catch (EJBException e) {
        }
    }

    /**
     * This test ensures that the system cannont remove an unknown object
     */
    @Test
    public void deleteUnknownCustomer() throws Exception {
        Customer customer;

        // Removes an object with a null parameter
        try {
            CustomerDelegate.deleteCustomer(null);
            fail("Object with null parameter should not be removed");
        } catch (EJBException e) {
        }
    }
}