# Ce contrôleur à deux responsabilités. Il offre l'action #login qui permet
# aux administrateurs de se connecter. Il implémente aussi les actions 
# #add_user, #list_users, et #delete_user utilisées pour maintenir la table
# des utilisateurs dans la base de données.
#
# Le LoginController partage sa mise en page avec AdminController
#
# Voir aussi: User

class LoginController < ApplicationController

  layout "admin"

  # Vous devez être connecté pour avoir accès à toutes ces fonctions
  # sauf #login
  before_filter :authorize, :except => :login

  # L'action par défaut affiche une page de statut.
  def index
    @total_orders   = Order.count
    @pending_orders = Order.count_pending
  end

  # Affiche le formulaire de connexion et attend que l'utilisateur
  # entre son nom et son mot de passe. Ces deux champs sont
  # ensuite validés et un objet user est ajouté à la session.
  def login
    if request.get?
      session[:user_id] = nil
      @user = User.new
    else
      @user = User.new(params[:user])
      logged_in_user = @user.try_to_login

      if logged_in_user
        session[:user_id] = logged_in_user.id
        redirect_to(:action => "index")
      else
        flash[:notice] = "Nom d'utilisateur ou mot de passe invalide"
      end
    end
  end
  
  # Ajouter un nouvel utilisateur à la base de données.
  def add_user
    if request.get?
      @user = User.new
    else
      @user = User.new(params[:user])
      if @user.save
        redirect_to_index("Utilisateur #{@user.name} créé")
      end
    end
  end

  # Supprime l'utilisateur avec l'ID spécifié dans la base de données.
  # Le modèle lève une exception si on essaye de supprimer le dernier
  # utilisateur.
  def delete_user
    id = params[:id]
    if id && user = User.find(id)
      begin
        user.destroy
        flash[:notice] = "Utilisateur #{user.name} supprimé"
      rescue
        flash[:notice] = "Impossible de supprimer cet utilisateur"
      end
    end
    redirect_to(:action => :list_users)
  end

  # Lister tous les utilisateurs.
  def list_users
    @all_users = User.find(:all)
  end

  # Se déconnecter en nettoyant l'entrée utilisateur dans les données
  # de session puis rediriger vers l'action #login.
  def logout
    session[:user_id] = nil
    flash[:notice] = "Déconnecté"
    redirect_to(:action => "login")
  end
end
