# Les fonctions d'administration permettent aux utilisateurs
# autorisés d'ajouter, de supprimer, de lister et de modifier 
# les produits. La classe initialement générée par l'échaffaudage
# Rails a ensuite été modifiée et il ne faut donc pas la regénérer.
#
# Seuls les adminstrateurs connectés peuvent utiliser les actions
# qui se trouvent ici. Voir # Application.authorize pour plus de
# détails
#
# Voir aussi: Product

class AdminController < ApplicationController

  before_filter :authorize

  # Un alias pour l'action #list. Présente la liste de tous les produits.
  def index
    list
    render_action 'list'
  end

  # Présente la liste de tous les produits.
  def list
    @product_pages, @products = paginate :product, :per_page => 10
  end

  # Montre les informations concernant un produit particulier.
  def show
    @product = Product.find(@params[:id])
  end

  # Première phase de la création d'un nouveau produit.
  # Le travail s'achève dans l'action #create.
  def new
    @product = Product.new
  end

  # Récupére les informations concernant un produit et
  # tente la création d'un nouvel enregistrement dans la
  # base de données.
  def create
    @product = Product.new(@params[:product])
    if @product.save
      flash['notice'] = 'Produit créé avec succès.'
      redirect_to :action => 'list'
    else
      render_action 'new'
    end
  end

  # Démarre la phase de modification d'un produit existant.
  # Le travail s'achève dans l'action #update.
  def edit
    @product = Product.find(@params[:id])
  end

  # Met à jour un produit existant sur les bases des valeurs
  # récupérées dans le formulaire.
  def update
    @product = Product.find(@params[:id])
    if @product.update_attributes(@params[:product])
      flash['notice'] = 'Produit modifié avec succès.'
      redirect_to :action => 'show', :id => @product
    else
      render_action 'edit'
    end
  end

  # Supprime le produit indiqué
  def destroy
    Product.find(@params[:id]).destroy
    redirect_to :action => 'list'
  end

  # Expédie un certain nombre de prduits. Cette action reboucle 
  # normalement sur elle-même. Elle commence par recherche les
  # commandes marquées comme devant être expédiées et les
  # envoie. Puis elle affiche la liste des commandes qui n'ont pas
  # encore été envoyées
  #
  # La vue contient une case à cocher pour chaque commande en 
  # attente d'expédition. Si l'utilisateur sélectionne la case de la 
  # commande dont l'id est 123, la méthode verra <tt>things_to_ship[123]</tt>
  # à "yes".
  def ship
    count = 0
    if things_to_ship = params[:to_be_shipped]
      count = do_shipping(things_to_ship)
      if count > 0
        count_text = pluralize(count, "commande")
        flash.now[:notice] = "#{count_text} marquée(s) comme expédiée(s)"
      end
    end
    @pending_orders = Order.pending_shipping
  end

  private

  def do_shipping(things_to_ship)
    count = 0
    things_to_ship.each do |order_id, do_it|
      if do_it == "yes"
        order = Order.find(order_id)
        order.mark_as_shipped
        order.save
        count += 1
      end
    end
    count
  end

  def pluralize(count, noun)
    case count
    when 0: "Aucun(e) #{noun.pluralize}"
    when 1: "Un(e) #{noun}"
    else    "#{count} #{noun.pluralize}"
    end
  end
end
