# Le StoreController gère le côté acheteur de la boutique.
#
# [#index]        Affiche le catalogue
# [#add_to_cart]  Ajoute le produit sélectionné au panier
# [#display_cart] Montre le contenu du panier
# [#empty_cart]   Vide le panier
# {#checkout}     Démarre la phase de confirmation de la commande
# [#save_order]   Finalise la confirmation en sauvegardant la commande
class StoreController < ApplicationController

  before_filter :find_cart, :except => :index

  # Affiche le catalogue: une liste de produits disponibles à la vente.
  def index
    @products = Product.salable_items
  end

  # Ajoute le produit sélectionné au panier.
  def add_to_cart
    product = Product.find(params[:id])
    @cart.add_product(product)
    redirect_to(:action => 'display_cart')
  rescue
    logger.error("Tentative d'accès à un produit invalide #{params[:id]}")
    redirect_to_index('Produit invalide')
  end
  
  # Affiche le contenu du panier. Si le panier est
  # vide, afficher un message d'information et retourner
  # au catalogue.
  def display_cart
    @items = @cart.items
    if @items.empty?
      redirect_to_index("Votre panier est actuellement vide")
    end

    if params[:context] == :checkout
      render(:layout => false)
    end
  end

  # Enlever tous les articles du panier
  def empty_cart
    @cart.empty!
    redirect_to_index('Votre panier est maintenant vide')
  end

  # Demander les informations de contact du client et le mode de paiement.
  # La procédure de conformation de la commande s'achève avec la
  # méthode #save_order.
  def checkout
    @items = @cart.items
    if @items.empty?
      redirect_to_index("Il n'y a rien dans votre panier!")
    else
      @order = Order.new
    end
  end

  # Appelé depuis la vue de confirmation. la panier est converti en
  # commande et sauvegarder dans la base de données.
  def save_order
    @order = Order.new(params[:order]) 
    @order.line_items << @cart.items      
    if @order.save                       
      @cart.empty!
      redirect_to_index('Merci pour cette commande.')
    else
      render(:action => 'checkout')          
    end
  end

  private

  # Sauvegarder l'objet panier dans la variable d'instance @cart variable.
  # Si le panier existe déjà dans le cache alors on l'assigne à cette variable
  #  sinon on en crée un nouveau et on l'ajoute à la session.
  def find_cart
    @cart = (session[:cart] ||= Cart.new)
  end
end
